<?php

namespace App\ExceptionHandler;

use Monolog\Formatter\JsonFormatter;

final class ExceptionLogger implements LogException
{
    private const TRACE_LIMIT = 5000;

    public function log(\Throwable $exception): void
    {
        $formatter = new JsonFormatter();

        $message = sprintf(
            'Error 500: file %s at line %s: %s',
            $exception->getFile(),
            $exception->getLine(),
            $exception->getMessage()
        );

        $dateTime = (new \DateTime('now', new \DateTimeZone('UTC')))->format(
            'Y-m-d H:i:s e'
        );

        $trace = $this->getTrace($exception);

        $log = $formatter->format([
            'message' => $message,
            'datetime' => $dateTime,
            'level_name' => 'ERROR',
            'context' => [
                'trace' => $trace,
                'request' => [
                    'post' => $_POST,
                    'get' => $_GET,
                    'files' => $_FILES,
                ]
            ],
        ]);

        log_message('error', $log);
    }

    private function getTrace(\Throwable $exception): array
    {
        $trace = $exception->getTrace();

        // last trace is always about system/core/CodeIgniter.php:289, it can be skipped
        array_pop($trace);

        if (strlen(print_r($trace, true)) > self::TRACE_LIMIT) {
            foreach($trace as $key => $traceValue) {
                if (isset($trace[$key]['args'], $trace[$key]['type'])) {
                    unset($trace[$key]['args'], $trace[$key]['type']);
                }
            }

            $trace = [
                'warning' => strtoupper('Args were removed from trace due to log size limit'),
                'trace' => $trace
            ];
        }

        return $trace;
    }
}
