<?php

namespace App\Integrations\Mft;

use App\Libraries\infrastructure\ScheduledJobLoggerService;
use phpseclib\Net\SFTP;

final class MftClient
{
    private ScheduledJobLoggerService $scheduledJobLoggerService;
    private string $logEntityName;

    /** @var SFTP MFT Connection */
    private $connection;

    public function __construct(ScheduledJobLoggerService $scheduledJobLoggerService, string $logEntityName)
    {
        $this->scheduledJobLoggerService = $scheduledJobLoggerService;
        $this->logEntityName = $logEntityName;
    }

    public function connect(): void
    {
        $connection = new SFTP(SWIFLOG_FTP);
        $connection->setTimeout(0);

        if (!$connection->login(SWIFLOG_FTP_USER, SWIFLOG_FTP_PWD)) {
            unset($connection);
            $this->scheduledJobLoggerService->log($this->logEntityName, 'MFT error - Cannot login');

            throw new \Exception($this->logEntityName, 'Cannot login into your server');
        }

        $this->connection = $connection;
    }

    public function getFileList(string $path, array $requiredExtensions): array
    {
        $this->connection->chdir($path);
        $fileList = $this->connection->nlist();

        if (!is_array($fileList)) {
            $this->scheduledJobLoggerService->log($this->logEntityName, 'MFT error - Cannot get list of files');

            throw new \Exception($this->logEntityName, 'Cannot get list of files');
        }

        foreach ($fileList as $key => $fileName) {
            if (in_array($fileName, ['.', '..'])) {
                unset($fileList[$key]);
                continue;
            }

            $extension = pathinfo($fileName, PATHINFO_EXTENSION);

            if (!in_array($extension, $requiredExtensions)) {
                unset($fileList[$key]);
            }
        }

        return $fileList;
    }

    public function filesize(string $path): string
    {
        return (string)$this->connection->filesize($path);
    }

    public function delete(string $path, bool $recursive = false): bool
    {
        return $this->connection->delete($path, $recursive);
    }

    public function get(string $sourcePath, string $destinationPath = '')
    {
        if ($destinationPath === '') {
            $destinationPath = false;
        }

        return $this->connection->get($sourcePath, $destinationPath);
    }

    public function put(string $destinationPath, $data): bool
    {
        return $this->connection->put($destinationPath, $data);
    }
}
