<?php

namespace App\Mobile\V1\Trips\Actions\Request;

use App\Mobile\V1\Trips\Actions\ActionRequest;

/**
 * Validate request
 */
class ActionRequestValidation
{
    /**
     * @param array $request
     * @param string $method
     * @return array of missing fields
     */
    public static function missingRequiredFields(array $request, string $method): array
    {
        $missingFields = [];

        switch ($method) {
            case ActionRequest::METHOD_CREATE:
                $fieldsToCheck = ActionRequestFields::BODY_FIELDS_CREATE['required'];
                break;
            case ActionRequest::METHOD_UPDATE:
                $fieldsToCheck = ActionRequestFields::BODY_FIELDS_UPDATE['required'];
                break;
            default:
                $fieldsToCheck = [];
                break;
        }

        if (count($request) > 0) {
            $validatingRequest = self::flattenRequest($request);

            if (isset($request['details']['position'])) {
                $fieldsToCheck[] = 'position_location_lat';
                $fieldsToCheck[] = 'position_location_long';
                $fieldsToCheck[] = 'position_date';
            }

            foreach ($fieldsToCheck as $fieldToCheck) {
                if (!self::arrayValueExists($validatingRequest, $fieldToCheck)) {
                    $missingFields[] = $fieldToCheck;
                }
            }
        }

        return $missingFields;
    }

    /**
     * @param array $request
     * @param string $method
     * @return array of invalid fields
     */
    public static function validateDataType($request, $method) : array
    {
        $invalidFields = [];

        switch ($method) {
            case ActionRequest::METHOD_CREATE:
                $fieldsToCheck = ActionRequestFields::BODY_FIELDS_CREATE['all'];
                break;
            case ActionRequest::METHOD_UPDATE:
                $fieldsToCheck = ActionRequestFields::BODY_FIELDS_UPDATE['all'];
                break;
            default:
                $fieldsToCheck = [];
                break;
        }

        $validatingRequest = self::flattenRequest($request);

        if (isset($request['details']['position'])) {
            $fieldsToCheck['position_location_lat'] = 'numeric';
            $fieldsToCheck['position_location_long'] = 'numeric';
            $fieldsToCheck['position_date'] = 'string';
        }

        foreach ($validatingRequest as $field => $value) {
            if (self::arrayKeyExists($fieldsToCheck, $field)) {
                $dataTypeValidation = "is_" . $fieldsToCheck[$field];

                if ($field == 'documents') {
                    $validType = $dataTypeValidation((array)$value);
                } elseif ($field == 'id') {
                    $validType = $value ? $dataTypeValidation((int)$value) : $dataTypeValidation($value);
                } else {
                    $validType = $dataTypeValidation($value);
                }

                //for any field: if $value has value and wrong datatype
                if (!$validType) {
                    $invalidFields[] = $field;
                }
            }
        }

        return $invalidFields;
    }

    /**
     * @param array $fields
     * @return string
     */
    public function formatValidatedFields(array $fields)
    {
        $list = '';

        foreach ($fields as $field) {
            $list .= str_replace('_', ' ', $field) . ', ';
        }

        return rtrim($list, ', ');
    }

    /**
     * @param array $array
     * @param string $keySearch
     * @return bool
     */
    private static function arrayKeyExists($array, $keySearch)
    {
        foreach ($array as $key => $item) {
            if ($key == $keySearch) {
                return true;
            } elseif (is_array($item) && self::arrayKeyExists($item, $keySearch)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param array $array
     * @param string $field
     * @return bool
     */
    private static function arrayValueExists($array, $field)
    {
        return array_key_exists($field, $array) && $array[$field] !== null;
    }

    /**
     * @param array $request
     * @return array
     */
    private static function flattenRequest($request)
    {
        $validatingStructure = [];

        foreach ($request as $key => $item) {
            if (is_array($item)) {
                foreach ($item as $k => $v) {
                    if ($k == 'position') {
                        continue;
                    }

                    $validatingStructure[$k] = $v;
                }
            } else {
                $validatingStructure[$key] = $item;
            }
        }

        if (isset($request['details']['position'])) {
            $validatingStructure['position_location_lat'] = $request['details']['position']['location']['lat'] ?? null;
            $validatingStructure['position_location_long'] = $request['details']['position']['location']['long'] ?? null;
            $validatingStructure['position_date'] = $request['details']['position']['date'] ?? null;
        }

        return $validatingStructure;
    }
}
