<?php

namespace App\Mobile\V1\Trips\Model;

use App\Mobile\V1\Trips\Actions\Enum\ActionType;

class Merchandise
{
    /**
     * @var int
     */
    private $id;
    /**
     * @var float
     */
    private $width;
    /**
     * @var float
     */
    private $height;
    /**
     * @var float
     */
    private $length;
    /**
     * @var float
     */
    private $weight;
    /**
     * @var string
     */
    private $orderNum;
    /**
     * @var float
     */
    private $quantity;
    /**
     * @var bool
     */
    private $isOnGround;
    /**
     * @var string
     */
    private $description;
    /**
     * @var bool
     */
    private $isDangerous;
    /**
     * @var bool
     */
    private $isStackable;
    /**
     * @var bool
     */
    private $isSplittable;

    /**
     * @param array $merchandise
     * @return static
     */
    public static function fromArray(array $merchandise): self
    {
        $merchandiseIntegerId = (int)$merchandise['id'];
        if (is_string($merchandise['id']) && intval($merchandiseIntegerId) !== $merchandiseIntegerId) {
            throw new \InvalidArgumentException("Merchandise id is not a string that contains integer only.");
        }
        return new Merchandise(
            $merchandiseIntegerId,
            $merchandise['width'],
            $merchandise['height'],
            $merchandise['length'],
            $merchandise['weight'],
            $merchandise['orderNum'],
            $merchandise['quantity'],
            self::makeBoolean($merchandise['isOnGround']),
            $merchandise['description'],
            self::makeBoolean($merchandise['isDangerous']),
            self::makeBoolean($merchandise['isStackable']),
            self::makeBoolean($merchandise['isSplittable'])
        );
    }

    /**
     * @param string $input
     * @return bool
     */
    private static function makeBoolean(string $input): bool
    {
        if ($input === '0') {
            return false;
        }
        if ($input === '1') {
            return true;
        }
        throw new \InvalidArgumentException("Unexpected string not translating to boolean.");
    }

    public function __construct(
        int $id,
        float $width,
        float $height,
        float $length,
        float $weight,
        string $orderNum,
        float $quantity,
        bool $isOnGround,
        string $description,
        bool $isDangerous,
        bool $isStackable,
        bool $isSplittable
    ) {
        $this->id = $id;
        $this->width = $width;
        $this->height = $height;
        $this->length = $length;
        $this->weight = $weight;
        $this->orderNum = $orderNum;
        $this->quantity = $quantity;
        $this->isOnGround = $isOnGround;
        $this->description = $description;
        $this->isDangerous = $isDangerous;
        $this->isStackable = $isStackable;
        $this->isSplittable = $isSplittable;
    }

    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'width' => $this->width,
            'height' => $this->height,
            'length' => $this->length,
            'weight' => $this->weight,
            'orderNum' => $this->orderNum,
            'quantity' => (int)$this->quantity,
            'isOnGround' => $this->isOnGround ? 1 : 0,
            'description' => $this->description,
            'isDangerous' => $this->isDangerous ? 1 : 0,
            'isStackable' => $this->isStackable ? 1 : 0,
            'isSplittable' => $this->isSplittable ? 1 : 0,
        ];
    }

    /**
     * @param \CI_DB_mysqli_driver $db
     * @param int $orderId
     * @return array|mixed
     */
    public static function getItems(\CI_DB_mysqli_driver $db, int $orderId)
    {
        $sql = "
                SELECT ocar.id,
                    o.id as orderId,
                    o.order_id as orderNum,
                    (ocar.weight * 1000) as weight,
                    ocar.length,
                    ocar.width,
                    ocar.height,
                    ocar.quantity,
                    IF(car.goods_description IS NULL, '', car.goods_description) as description,
                    car.dg_goods as isDangerous,
                    car.stackable as isStackable,
                    car.grounded as isOnGround,
                    car.splittable as isSplittable
                FROM tb_order_cargodetails ocar
                    JOIN tb_cargo_details car ON ocar.cargo_id=car.id
                    JOIN tb_orders o ON ocar.order_id=o.id
                WHERE ocar.order_id = ?";

        $result = $db->query($sql, [$orderId])->result_array();

        return is_array($result) ? $result : [];
    }

    /**
     * @return array
     */
    public static function getValidActionTypes()
    {
        return [ActionType::PICKUP, ActionType::DELIVERY];
    }
}
