<?php

namespace App\Mobile\V1\Trips\Model;

use App\Mobile\V1\Trips\Actions\Enum\ActionType;
use App\Mobile\V1\Trips\Exception\ActionTypeNotValidException;
use Carbon\Carbon;

class StopAction
{
    /**
     * @var int|null
     */
    private $id;
    /**
     * @var string
     */
    private $virtual_id;
    /**
     * @var string
     */
    private $type;
    /**
     * @var string
     */
    private $instructions;
    /**
     * @var Carbon|null
     */
    private $minDateTime;
    /**
     * @var Carbon|null
     */
    private $maxDateTime;
    /**
     * @var array
     */
    private $merchandise;
    /**
     * @var StopActionDetails|null
     */
    private $details;

    public function __construct(
        ?int $id,
        string $virtualId,
        string $type,
        string $instructions,
        ?Carbon $minDateTime,
        ?Carbon $maxDateTime,
        StopActionDetails $details,
        Merchandise ...$merchandise
    ) {
        $this->id = $id;
        $this->virtual_id = $virtualId;
        $this->type = $type;
        $this->instructions = $instructions;
        $this->minDateTime = $minDateTime;
        $this->maxDateTime = $maxDateTime;
        $this->merchandise = $merchandise;
        $this->details = $details;
    }

    public static function fromArray(array $array, string $curtz): self
    {
        if ($array['type'] === '0' || $array['type'] === 'zero') {
            throw new ActionTypeNotValidException();
        }
        $type = $array['type'];

        return new StopAction(
            $array['id'],
            $array['virtual_id'],
            $type,
            $array['instructions'],
            (new Carbon($array['minDateTime']))->tz($curtz),
            (new Carbon($array['maxDateTime']))->tz($curtz),
            StopActionDetails::fromArray($array['details'], $curtz),
            ...array_map(function (array $merchandise): Merchandise {
                return Merchandise::fromArray($merchandise);
            }, $array['merchandise'])
        );
    }

    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'virtual_id' => $this->virtual_id,
            'type' => $this->type,
            'instructions' => $this->instructions,
            'minDateTime' => is_null($this->minDateTime) ? null : $this->minDateTime->toDateTime()->format("Y-m-d\TH:iO"),
            'maxDateTime' => is_null($this->maxDateTime) ? null : $this->maxDateTime->toDateTime()->format("Y-m-d\TH:iO"),
            'merchandise' => array_map(function (Merchandise $merchandise) {
                return $merchandise->toArray();
            }, $this->merchandise),
            'details' => is_null($this->details) ? null : $this->details->toArray(),
        ];
    }

    public function isEqual(string $actionTypeToCompare): bool
    {
        return $this->type == $actionTypeToCompare;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function isGateIn(): bool
    {
        return $this->type === ActionType::GATE_IN;
    }

    public function isGateOut(): bool
    {
        return $this->type === ActionType::GATE_OUT;
    }

    public function getStatusCode(): ?string
    {
        return $this->details->getStatusCode();
    }

    public function getDateTime(): ?Carbon
    {
        return $this->details->getDate();
    }

    public function getDetails(): StopActionDetails
    {
        return $this->details;
    }
}
