<?php

namespace App\MobileEvents;

use App\EventsQueue\EventQueueManager;
use App\MobileEvents\Handler\InvalidPayloadException;
use App\MobileEvents\Handler\MobileEventHandler;
use App\MobileEvents\Handler\MobileEventHandlerSpecificException;
use CI_Benchmark;
use CI_Controller;
use Throwable;

final class EventManager extends MobileEventType
{
    /**
     * @var CI_Controller
     */
    private $CI;

    /**
     * @var CI_Benchmark
     */
    private $benchmark;

    /**
     * @var EventQueueManager
     */
    private $eventQueueManager;

    public function __construct(CI_Controller $CI, EventQueueManager $eventQueueManager)
    {
        $this->CI = $CI;
        $this->benchmark = $this->CI->benchmark;
        $this->eventQueueManager = $eventQueueManager;
    }

    public function lockAndProcessEvents(): void
    {
        $eventToProcess = $this->eventQueueManager->getNotStartedEvents($this->getEventType());
        if ($eventToProcess->num_rows() === 0) {
            return;
        }

        $this->eventQueueManager->markAsPending(array_column($eventToProcess->result(), 'id'));
        $this->processEvents($eventToProcess->result());
    }

    public function lockAndProcessNotFinishedEvents(): void
    {
        $eventToProcess = $this->eventQueueManager->getNotFinishedEvents($this->getEventType());
        if ($eventToProcess->num_rows() === 0) {
            return;
        }

        $this->eventQueueManager->markAsPending(array_column($eventToProcess->result(), 'id'));
        $this->processEvents($eventToProcess->result());
    }

    private function processEvents(array $mobileEvents): void
    {
        foreach ($mobileEvents as $mobileEvent) {
            try {
                $this->benchmark->mark('event_start');
                $this->eventQueueManager->markAsStarted($mobileEvent->id);

                echo sprintf('Processing event: %s' . PHP_EOL, $mobileEvent->id);

                $mobileEventHandler = new MobileEventHandler($this->CI, $mobileEvent->event_name);
                $payload = unserialize($mobileEvent->payload);

                try {
                    $mobileEventHandler->handle($payload);
                    $this->eventQueueManager->markAsComplete($mobileEvent->id);
                    echo sprintf('Completed event: %s' . PHP_EOL, $mobileEvent->id);
                } catch (InvalidPayloadException $exception) {
                    $this->eventQueueManager->markAsFailed($mobileEvent->id, 'Handler was not found');
                } catch (MobileEventHandlerSpecificException $exception) {
                    $this->eventQueueManager->markAsFailed($mobileEvent->id, $exception->getMessage());
                }
            } catch (Throwable $ex) {
                $this->eventQueueManager->markAsFailed($mobileEvent->id, $ex->getMessage());
                echo sprintf('Failed processing event: %s' . PHP_EOL, $mobileEvent->id);
            } finally {
                $this->benchmark->mark('event_end');
                $timeTakenByJob = $this->benchmark->elapsed_time('event_start', 'event_end');
                $this->eventQueueManager->setTimeTaken($mobileEvent->id, $timeTakenByJob);
            }
        }
    }
}
