<?php

namespace App\MobileEvents\Payload\Builder;

use App\MobileEvents\Payload\Document\UploadDocument;
use App\MobileEvents\Payload\Position\OptionalPosition;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use App\MobileEvents\Payload\UploadCustomsPayload;
use Exception;
use DateTimeZone;

final class UploadCustomsPayloadBuilder
{
    private $position;
    private $tripIdentity;
    private $document;
    private $currentTimeZone;
    private $podType;
    private $userId;
    private $offset;
    private $createdSource;
    private $order;

    public function __construct()
    {
        $this->position = new OptionalPosition('', '');
    }

    /**
     * @throws Exception
     */
    public function build(): UploadCustomsPayload
    {
        try {
            $currentTimeZone = new DateTimeZone($this->currentTimeZone);
            return new UploadCustomsPayload(
                $this->position,
                $this->tripIdentity,
                $this->document,
                $currentTimeZone,
                $this->userId,
                $this->offset,
                $this->podType,
                $this->createdSource,
                $this->order
            );
        } catch (\Throwable $error) {
            throw new \Exception(sprintf('UploadCustomsPayload cannot be build. %s', $error->getMessage()));
        }
    }

    /**
     * @throws Exception
     */
    public function position(
        string $latitude,
        string $longitude
    ): UploadCustomsPayloadBuilder {
        $this->position = new OptionalPosition($latitude, $longitude);
        return $this;
    }

    /**
     * @throws Exception
     */
    public function trip(
        string $shipmentId,
        string $tripId,
        string $driverId,
        string $orderId
    ): UploadCustomsPayloadBuilder {
        $this->tripIdentity = new RequiredTripIdentity($orderId, '', $driverId, $shipmentId, $tripId);
        return $this;
    }

    /**
     * @throws Exception
     */
    public function document(
        string $fileName,
        string $fileType
    ): UploadCustomsPayloadBuilder {
        $this->document = new UploadDocument($fileName, $fileType, '', '');
        return $this;
    }

    public function uploadCustomsPayload(
        string $podType,
        string $currentTimeZone,
        string $offset,
        string $createdSource,
        string $order,
        string $userId
    ): UploadCustomsPayloadBuilder {
        $this->podType = $podType;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;
        $this->createdSource = $createdSource;
        $this->order = $order;
        $this->userId = $userId;
        return $this;
    }
}