<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeZone;
use Exception;

final class DriverCollectionPayload extends AbstractEdiPayload implements EventPayload
{
    const SVKONEKT_QUOTE_STATUS_CODE = '0212';

    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var string */
    private $createdSource;

    /** @var string */
    private $userId;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $offset;

    /** @var string */
    private $legOrderId;

    /** @var string */
    private $stopLegId;

    /**
     * @param RequiredTripIdentity $tripIdentity
     * @param Position $position
     * @param string $createdSource
     * @param string $userId
     * @param DateTimeZone $currentTimeZone
     * @param string $offset
     * @param string $legOrderId
     * @param string $stopLegId
     * 
     * @throws Exception
     */
    public function __construct(RequiredTripIdentity $tripIdentity, Position $position, string $createdSource, string $userId, DateTimeZone $currentTimeZone, string $offset, string $legOrderId, string $stopLegId)
    {
        $this->tripIdentity = $tripIdentity;
        $this->position = $position;
        $this->createdSource = $createdSource;
        $this->userId = $userId;
        $this->currentTimeZone = $currentTimeZone;
        $this->offset = $offset;
        $this->legOrderId = $legOrderId;
        $this->stopLegId = $stopLegId;

        if ($this->validatePayload() === false) {
            throw new Exception('Reject trip is invalid');
        }
    }

    /**
     * Return payload internals into the array in format expected by external systems
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            "shipment_id" => $this->getTripIdentity()->getShiftId(),
            "trip_id" => $this->getTripIdentity()->getTripId(),
            "driver_id" => $this->getTripIdentity()->getDriverId(),
            "order_id" => $this->getLegOrderId(),
            "stop_id" => $this->getStopLegId(),
            "latitude" => $this->getPosition()->getLatitude(),
            "longitude" => $this->getPosition()->getLongitude(),
            "curtz" => $this->getCurrentTimeZone()->getName(),
            "hrs" => $this->getOffset()
        ];
    }

    private function validatePayload(): bool
    {
        return !empty($this->getUserId()) &&
            !empty($this->getCurrentTimeZone()) &&
            !empty($this->getOffset()) &&
            !empty($this->getLegOrderId());
    }

    /**
     * @return string
     */
    public function getCreatedSource(): string
    {
        return $this->createdSource;
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->userId;
    }

    /**
     * @return DateTimeZone
     */
    public function getCurrentTimeZone(): DateTimeZone
    {
        return $this->currentTimeZone;
    }

    /**
     * @return string
     */
    public function getOffset(): string
    {
        return $this->offset;
    }

    /**
     * @return RequiredTripIdentity
     */
    public function getTripIdentity(): RequiredTripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }

    /**
     * @return string
     */
    public function getLegOrderId(): string
    {
        return $this->legOrderId;
    }

    /**
     * @return string
     */
    public function getStopLegId(): string
    {
        return $this->stopLegId;
    }

    public function getDataToSvkonektQuote(): array
    {
        if (!$this->isSvkonektQuotePayload()) {
            return [];
        }

        $dataToReturn = $this->toArray();
        $dataToReturn['user_id'] = $this->getUserId();
        $dataToReturn['ord_id'] = $this->getTripIdentity()->getOrderId();
        $dataToReturn['status_code'] = self::SVKONEKT_QUOTE_STATUS_CODE;

        return $dataToReturn;
    }
}