<?php

namespace App\MobileEvents\Payload;

use App\EventsQueue\Payload\EventPayload;
use App\MobileEvents\Payload\Document\UploadDocument;
use App\MobileEvents\Payload\Position\Position;
use App\MobileEvents\Payload\Trip\RequiredTripIdentity;
use DateTimeZone;
use Exception;

final class UploadCustomsPayload implements EventPayload
{
    /** @var Position */
    private $position;

    /** @var RequiredTripIdentity */
    private $tripIdentity;

    /** @var UploadDocument */
    private $document;

    /** @var DateTimeZone */
    private $currentTimeZone;

    /** @var string */
    private $userId;

    /** @var string */
    private $offset;

    /** @var string */
    private $podType;

    /** @var string */
    private $createdSource;

    /** @var string */
    private $order;

    /**
     * @param Position $position
     * @param RequiredTripIdentity $tripIdentity
     * @param UploadDocument $document
     * @param DateTimeZone $currentTimeZone
     * @param string $userId
     * @param string $offset
     * @param string $podType
     * @param string $createdSource
     * @param string $order
     * @throws Exception
     */
    public function __construct(
        Position $position,
        RequiredTripIdentity $tripIdentity,
        UploadDocument $document,
        DateTimeZone $currentTimeZone,
        string $userId,
        string $offset,
        string $podType,
        string $createdSource,
        string $order
    ) {
        $this->position = $position;
        $this->tripIdentity = $tripIdentity;
        $this->document = $document;
        $this->currentTimeZone = $currentTimeZone;
        $this->userId = $userId;
        $this->offset = $offset;
        $this->podType = $podType;
        $this->createdSource = $createdSource;
        $this->order = $order;

        if ($this->validatePayload() === false) {
            throw new Exception('Upload customs payload is invalid');
        }
    }

    public function toArray(): array
    {
        return array(
            "shipment_id" => $this->getTripIdentity()->getShiftId(),
            "trip_id" => $this->getTripIdentity()->getTripId(),
            "driver_id" => $this->getTripIdentity()->getDriverId(),
            "order_id" => $this->getTripIdentity()->getOrderId(),
            "file_name" => $this->getDocument()->getFileName(),
            "file_type" => $this->getDocument()->getFileType(),
            'pod_type' => $this->getPodType(),
            'user_id' => $this->getUserId(),
            "latitude" => $this->getPosition()->getLatitude(),
            "longitude" => $this->getPosition()->getLongitude(),
            "curtz" => $this->getCurrentTimeZone()->getName(),
            "hrs" => $this->getOffset(),
        );
    }

    private function validatePayload(): bool
    {
        return !empty($this->getPodType()) &&
            !empty($this->getUserId());
    }

    /**
     * @return string
     */
    public function getPodType(): string
    {
        return $this->podType;
    }

    /**
     * @return string
     */
    public function getUserId(): string
    {
        return $this->userId;
    }

    /**
     * @return Position
     */
    public function getPosition(): Position
    {
        return $this->position;
    }

    /**
     * @return RequiredTripIdentity
     */
    public function getTripIdentity(): RequiredTripIdentity
    {
        return $this->tripIdentity;
    }

    /**
     * @return DateTimeZone
     */
    public function getCurrentTimeZone(): DateTimeZone
    {
        return $this->currentTimeZone;
    }

    /**
     * @return string
     */
    public function getOffset(): string
    {
        return $this->offset;
    }

    /**
     * @return UploadDocument
     */
    public function getDocument(): UploadDocument
    {
        return $this->document;
    }

    /**
     * @return string
     */
    public function getCreatedSource(): string
    {
        return $this->createdSource;
    }

    /**
     * @return string
     */
    public function getOrder(): string
    {
        return $this->order;
    }
}