<?php

namespace App\Pagination;

class DataTablesPaginationBuilder implements PaginationBuilderInterface
{
    private const PAGINATION_OPTIONS = [10, 20, 30, 40, 50, 'All'];
    /** @var int */
    private $currentPage;
    /** @var bool */
    private $hasNextPage;
    /** @var bool */
    private $hasPreviousPage;
    /** @var int */
    private $limit;
    /** @var array */
    private $urlArguments;

    public function __construct(
        int $limit = 10,
        int $currentPage = 1,
        bool $hasPreviousPage = false,
        bool $hasNextPage = false,
        array $urlArguments = []
    ) {
        $this->limit = $limit;
        $this->currentPage = $currentPage;
        $this->hasPreviousPage = $hasPreviousPage;
        $this->hasNextPage = $hasNextPage;
        $this->urlArguments = $urlArguments;
    }

    public function renderHtml(): string
    {
        return sprintf(
            '<div id="custom-paginate" style="display:flex; flex-direction: row; justify-content: flex-end;">
                <div class="paginate-length">
                    <label>
                        <select
                            %s
                            class="form-control"
                            id="pagination-length"
                            style="width: 75px; margin-right: 15px;"
                        >
                            %s
                        </select>
                    </label>
                </div>
                <div class="paginate-buttons">
                    <ul class="pagination" style="padding: 0; margin: 0;">
                        %s
                    </ul>
                </div>
            </div>',
            $this->buildOnchangePagination(),
            $this->buildOptionList(),
            $this->buildButtons()
        );
    }

    private function buildOnchangePagination(): string
    {
        return sprintf(
            'onchange="location = \'?limit=\'+this.options[this.selectedIndex].value+\'&%s\'"',
            $this->resolveArguments()
        );
    }

    private function buildOptionList(): string
    {
        $options = [];
        foreach (self::PAGINATION_OPTIONS as $option) {
            if ($this->limit === 0 && $option === 'All') {
                $options[] = sprintf(
                    '<option value="%s" selected>%s</option>', $option, $option
                );
                continue;
            }
            if ($this->limit === $option) {
                $options[] = sprintf('<option value="%s" selected>%s</option>', $option, $option);
                continue;
            }
            $options[] = sprintf('<option value="%s">%s</option>', $option, $option);
        }

        return implode('', $options);
    }

    private function buildButtons(): string
    {
        return implode('', [$this->buildPreviousButton(), $this->buildNextButton()]);
    }


    private function resolveArguments(): string
    {
        $arguments = $this->urlArguments;

        unset($arguments['page'], $arguments['limit']);
        return implode(
            '&',
            array_map(static function ($key, $value) {
                if (is_array($value)) {
                    return implode(
                        '&',
                        array_map(static function ($subValue) use ($key) {
                            return sprintf('%s=%s', $key . '[]', $subValue);
                        }, $value)
                    );
                }
                return sprintf('%s=%s', $key, $value);
            }, array_keys($arguments), $arguments)
        );
    }

    private function buildPreviousButton(): string
    {
        if (!$this->hasPreviousPage) {
            return '<li class="paginate_button previous disabled" tabindex="0"><a href="#">Previous</a></li>';
        }

        return sprintf(
            '<li class="paginate_button previous" tabindex="0"><a href="?page=%d&limit=%d&%s">Previous</a></li>',
            $this->currentPage - 1,
            $this->limit,
            $this->resolveArguments()
        );
    }

    private function buildNextButton(): string
    {
        if (!$this->hasNextPage) {
            return '<li class="paginate_button next disabled" tabindex="0"><a href="#">Next</a></li>';
        }

        return sprintf(
            '<li class="paginate_button next" tabindex="0"><a href="?page=%d&limit=%d&%s">Next</a></li>',
            $this->currentPage + 1,
            $this->limit,
            $this->resolveArguments()
        );
    }
}
