<?php

namespace App\PodUploads\Builders;

use App\PodUploads\MakePod\CompletedOrderPodData;
use App\PodUploads\MakePod\PodData;
use CI_Controller;
use Exception;
use Mpdf\Mpdf;

class CompletedOrderEPodBuilder extends EPodBuilder {

    protected string $epodTemplate = 'admin/downloadCompletedOrder';
    private array $pickupComments = [];
    private string $referenceNumbers = '';
    private array $podUploads = [];
    private int $totalCargo = 0;

    public function __construct(string $order_id, CI_Controller $CI) {
        parent::__construct($order_id, $CI);

        $this->CI->load->model("common");

        $this->fetchReferences()
            ->fetchTotalCargo()
            ->fetchPodUploads();
    }

    /**
     * @return CompletedOrderEPodBuilder
     */
    public function createPdf():self {
        try {
            $pdf = new Mpdf([
                'tempDir' => FCPATH .'assets/tmp',
                'autoScriptToLang' => true,
                'autoLangToFont' => true
            ]);
            $pdf->WriteHTML($this->getHtml());
            $pdf->Output('epod' . $this->order->order_id .'.pdf', 'D');

        } catch (Exception $e) {
            log_message('error', 'downloadCompletedOrder error '.$e->getMessage());

            echo $this->getHtml();
        }

        return $this;
    }

    protected function getPodData (): PodData {
        $podData = parent::getPodData();
        $completedOrderPodData = CompletedOrderPodData::fromPodData($podData);

        $completedOrderPodData->add([
            'pickupComments' => implode('<br />', $this->pickupComments),
            'userid' => $this->CI->session->userdata('user_id') ?? '',
            'referenceNumbers' => $this->referenceNumbers,
            'totalCargo' => $this->totalCargo,
            'epods' => $this->podUploads
        ]);

        return $completedOrderPodData;
    }

    protected function parseStopStatusRow(object $row): void {
        parent::parseStopStatusRow($row);

        if ($row->comment !== '' && $row->status_code === '0500') {
            $this->pickupComments [] = trim($row->comment ?? "");
        }
    }

    protected function logCompleted():void {
        log_message(
            "error",
            "CompletedOrderEPodBuilder: epod generation completed for order :".
            " {$this->order->id} / {$this->order->order_id} / {$this->order->external_order_id}"
        );
    }

    private function fetchTotalCargo():self {
        $cargos = $this->CI->common->gettblrowdata([
            'order_id' => $this->order->id,
            'status'=>1
        ], 'sum(quantity) as total', 'tb_order_cargodetails', 0, 0);

        if(!empty($cargos)){
            $this->totalCargo = (int) $cargos['total'];
        }

        return $this;
    }

    private function fetchReferences():self {
        $references = $this->CI->common->getallorderreferences($this->order->id);

        if ($references->num_rows()) {
            $this->referenceNumbers = $references->row()->reference_id ?: '-';
        }

        return $this;
    }

    private function fetchPodUploads():self {
        $this->CI->load->model('PodUpload');

        $this->podUploads = $this->CI->PodUpload->getPodUploadsWithDetailsByOrderId($this->order->id);

        return $this;
    }
}
