<?php

declare(strict_types=1);

namespace App\Reporting\MisReport;

use App\Order\Constants\OrderSources;
use App\Reporting\MisReport\Vendor\VendorInterface;
use DateTimeImmutable;

final class PickupAttempts
{
    private const DATE_FORMAT = 'Y-m-d';

    private int $totalAttempts = 0;
    private int $totalReturnDispatchAttempts = 0;
    private $firstPickup;
    private $firstReturnDispatch;
    private $lastPickup;


    public function __construct(int $orderSource, array $orders, VendorInterface $vendor)
    {
        foreach ($orders as $order) {
            if ($order['scan_status_code'] === "006" || $order['scan_status_code'] === "1224") {
                $this->totalAttempts++;
                $this->totalReturnDispatchAttempts++;

                $this->updatePickupAttempt($order['scan_status_time']);

                $attemptDate = new DateTimeImmutable($order['scan_status_time']);

                if (!$this->firstReturnDispatch || $this->isEarlierThanFirst($attemptDate)) {
                    $this->firstReturnDispatch = $attemptDate;
                }
            }

            if ($orderSource === OrderSources::SWIFTLOG_RETURN && $order['scan_status_code'] === 'X-DDD3FP') {
                $this->totalAttempts++;

                $this->updatePickupAttempt($order['scan_status_time']);
            }

            if (in_array($order['scan_val'], ["Dispatched", "Out for delivery", "Out for Return"]) &&
                in_array($order['scan_instruction'], ["Out for delivery", "Out for pickup"])
                && $vendor->isDelhivery()) {
                $this->totalAttempts++;
                $this->totalReturnDispatchAttempts++;

                $this->updatePickupAttempt($order['scan_status_time']);

                $attemptDate = new DateTimeImmutable($order['scan_status_time']);

                if (!$this->firstReturnDispatch || $this->isEarlierThanFirst($attemptDate)) {
                    $this->firstReturnDispatch = $attemptDate;
                }
            }
        }
    }

    public function first(): string
    {
        if (!$this->firstPickup instanceof DateTimeImmutable) {
            return '';
        }

        return $this->firstPickup->format(self::DATE_FORMAT);
    }

    public function firstReturnDispatch(): string
    {
        if (!$this->firstReturnDispatch instanceof DateTimeImmutable) {
            return '';
        }

        return $this->firstReturnDispatch->format(self::DATE_FORMAT);
    }

    public function last(): string
    {
        if (!$this->lastPickup instanceof DateTimeImmutable) {
            return '';
        }

        return $this->lastPickup->format(self::DATE_FORMAT);
    }

    public function inTotal(): int
    {
        return $this->totalAttempts;
    }

    public function inTotalReturnDispatch(): int
    {
        return $this->totalReturnDispatchAttempts;
    }

    private function isEarlierThanFirst(DateTimeImmutable $attemptDate): bool
    {
        return $attemptDate < $this->firstPickup;
    }

    private function isLaterThanLast(DateTimeImmutable $attemptDate): bool
    {
        return $attemptDate > $this->lastPickup;
    }

    private function updatePickupAttempt(string $scanStatusTime)
    {
        $attemptDate = new DateTimeImmutable($scanStatusTime);

        if (!$this->firstPickup || $this->isEarlierThanFirst($attemptDate)) {
            $this->firstPickup = $attemptDate;
        }

        if (!$this->lastPickup || $this->isLaterThanLast($attemptDate)) {
            $this->lastPickup = $attemptDate;
        }
    }
}
