<?php

namespace App\ShipmentStops;

use App\ApiRequest\GetPartyInformation;
use App\ShiftsData\TbStopStatusFields\Status;
use App\ShiftsData\TbStopStatusFields\StatusId;
use App\ShiftsData\TbStopStatusFields\StopType;

class GetShipmentStopsService
{
    /**
     * @param string $shiftId
     * @param string $curtz
     * @param \CI_DB_mysqli_driver $db
     * @return array
     * @throws ShipmentOrderStopsNotFoundException
     */
    public static function getShipmentStops(string $shiftId, string $curtz, \CI_DB_mysqli_driver $db): array
    {
        //log_message("error","Innnnnnnnnn:::");
        $stops = (self::getStops($curtz, $shiftId, $db))->result() ?? [];
        //log_message("error","getStopsQryyyy:".$db->last_query());
        if (empty($stops)) {
            throw new ShipmentOrderStopsNotFoundException("Shipment order stops not found.");
        }

        $result = [];
        $location = [];

        $prevsts = "N";
        $place = "";

        foreach ($stops as $i => $stop) {
            $stopDetails = self::stopDetails($stop, $db);

            if (!$stopDetails) {
                continue;
            }

            $prevsts = ($prevsts == "C") ? "S" : "N";
            $status = $prevsts;

            $countPickupStops = 0;
            $countDropStops = 0;
            
            $bookingIds = [];

            foreach ($stopDetails as $stopDetail) {
                $bookingIds = [];
                $countPickupStops += (int)$stopDetail->cnt_stop_ids;
                $countDropStops += (int)$stopDetail->cnt_drop_stop_ids;

                if ($stopDetail->order_id) {
                    $bookingIds[] = $stopDetail->order_id;
                }
                //log_message("error","bookingIds:".json_encode($bookingIds));

                $knTruckingNumbers = self::getTruckingNumbers($db, $bookingIds, $stop->kn_tracking_no);
                $countStopDetails = $countPickupStops + $countDropStops;
                $countGateOut = self::countGateOutDetails($countPickupStops, $stop, $db);
                $countStatus = self::countStatuses($stop, $db);
    
                if ($countGateOut === $countStopDetails) {
                    $status = $prevsts = "C";
                } elseif ($countStatus > 0) {
                    $status = $prevsts = "S";
                }
    
                $status = ($status == "N" && $stop->stoptype==="P") ? "S" : $status;
                $orders = self::getOrderDetails($bookingIds, $db);
    
                list($place, $location, $destinationParty, $sourceParty, $destination, $source) = self::processAddressesInGetShipmentStops(
                    $orders,
                    $stop->stoptype,
                    $place,
                    $location,
                    $stop->id,
                    $shiftId,
                    $i,
                    $db
                );
    
                $result[] = PrepareShipmentStopResult::prepareShipmentStopResultWithPlaces(
                    $stop,
                    $status,
                    $bookingIds,
                    $place,
                    $location,
                    $source,
                    $destination,
                    $sourceParty,
                    $destinationParty,
                    $stop->ordernumber,
                    $knTruckingNumbers
                );
            }
            //log_message("error","result():::".json_encode($result));
            }
            
            
        return $result;
    }

    public static function getStops(string $curtz, int $shiftId, \CI_DB_mysqli_driver $db) : \CI_DB_mysqli_result
    {
        $sql = "SELECT
                st.id, st.stopname, st.plat, st.plng, st.stopcity, st.stoptype, st.ordernumber,
                convertToClientTZ(IF(st.stoptype = 'D', MIN(o.delivery_datetime), MIN(o.pickup_datetime)), ?) AS startdate,
                convertToClientTZ(IF(st.stoptype = 'D', MAX(o.drop_endtime), MAX(o.pickup_endtime)), ?) AS enddate,
                IFNULL(
					(SELECT SUM(cargo.weight) AS weight FROM tb_order_cargodetails cargo JOIN tb_orders odr ON odr.id=cargo.order_id WHERE cargo.status = 1 AND odr.shift_id=st.shipment_id), st.weight
				) AS weight,
				IFNULL(
					(SELECT SUM(cargo.volume) AS volume FROM tb_order_cargodetails cargo JOIN tb_orders odr ON odr.id=cargo.order_id WHERE cargo.status = 1 AND odr.shift_id=st.shipment_id), st.volume
				) AS volume,
                st.shipment_id AS shift_id,
                IF(o.created_source = 5, o_ref.ref_value, '') as kn_tracking_no
            FROM tb_shiporder_stops st
            LEFT JOIN tb_orders o ON o.shift_id = st.shipment_id
            LEFT JOIN tb_order_references o_ref ON o.id = o_ref.order_id
                AND o_ref.reference_id = 'XSR' -- XSR is a Tracking Number for SALOG
            WHERE st.shipment_id = ?
            GROUP BY st.id, st.ordernumber
            ORDER BY st.ordernumber";
        
        return $db->query($sql, [$curtz, $curtz, $shiftId]);
    }

    /**
     * @param $orders
     * @param $orderShipType
     * @param $place
     * @param $location
     * @param $stopId
     * @param $shiftId
     * @param int $index
     * @param \CI_DB_mysqli_driver $db
     * @return array
     */
    private static function processAddressesInGetShipmentStops(
        $orders,
        $orderShipType,
        $place,
        $location,
        $stopId,
        $shiftId,
        int $index,
        \CI_DB_mysqli_driver $db
    ): array {
        list($place, $location, $destinationParty, $sourceParty, $destination, $source, $stopsarr) = self::processLocationsInGetShipmentStops(
            $orders,
            $orderShipType,
            $place,
            $location,
            $stopId,
            $shiftId
        );
        $partyaddr = GetPartyInformation::getPartyAddrByOrdStops($stopsarr, $index, $db);
        if (!empty($partyaddr)) {
            if (!empty($partyaddr['source'])) {
                $source = $partyaddr['source'];
                $sourceParty = $partyaddr['source_party'];
            }
            if (!empty($partyaddr['destination'])) {
                $destination = $partyaddr['destination'];
                $destinationParty = $partyaddr['dest_party'];
            }
        }

        return array($place, $location, $destinationParty, $sourceParty, $destination, $source);
    }

    private static function processLocationsInGetShipmentStops(
        array $orders,
        $orderShipType,
        $place,
        array $location,
        $stopId,
        $shiftId
    ): array {
        $companyCode = "";

        if (empty($orders)) {
            $location = [
                'address' => '',
                'city' => '',
                'zip' => '',
                'country' => ''
            ];

            $place = '';
        }

        foreach ($orders as $order) {
            $order = (object)$order;
            $companyCode = $order->company_code;

            if ($orderShipType == "P") {
                $place = $order->pickup_company;
                $address = $order->pickup_address1;
                $city = $order->pickup_city;
                $zip = $order->pickup_pincode;
                $country = $order->pickup_country;
            } elseif ($orderShipType == "D") {
                $place = $order->delivery_company;
                $address = $order->delivery_address1;
                $city = $order->delivery_city;
                $zip = $order->delivery_pincode;
                $country = $order->delivery_country;
            }

            $location = [
                'address' => $address,
                'city' => $city,
                'zip' => $zip,
                'country' => $country
            ];
        }

        $source = $destination = $sourceParty = $destinationParty = "";

        $stopsarr = [
            'sstop' => ($orderShipType == "P") ? $stopId : 0,
            'dstop' => ($orderShipType == "D") ? $stopId : 0,
            'shipment_id' => $shiftId,
            'cmpcode' => $companyCode
        ];

        return [
            $place,
            $location,
            $destinationParty,
            $sourceParty,
            $destination,
            $source,
            $stopsarr
        ];
    }

    /**
     * @param \stdClass $stop
     * @param \CI_DB_mysqli_driver $db
     * @return \stdClass[]
     */
    private static function stopDetails(\stdClass $stop, \CI_DB_mysqli_driver $db)
    {
        $sql = "
                SELECT order_id, IF(stop_id = ?, COUNT(*), 0) AS cnt_stop_ids, IF(drop_stopid = ?, COUNT(*), 0) AS cnt_drop_stop_ids
                FROM tb_employee
                WHERE (stop_id = ? OR drop_stopid = ?) AND shift_id = ? AND `status` = ?
                GROUP BY order_id
              ";

        $params = [$stop->id, $stop->id, $stop->id, $stop->id, $stop->shift_id, 1];

        return $db->query($sql, $params)->result();
    }

    /**
     * @param int $countStops
     * @param \stdClass $stop
     * @param \CI_DB_mysqli_driver $db
     * @return int
     */
    private static function countGateOutDetails(int $countStops, \stdClass $stop, \CI_DB_mysqli_driver $db) : int
    {
        $db->from('tb_stop_status');
        $db->where("stop_id", $stop->id);
        $db->where("shipment_id", $stop->shift_id);
        $db->where("status", Status::ACTIVE);

        if ($countStops > 0) {
            $db->where("status_id", StatusId::PICKUP_GATE_OUT);
            $db->where("stop_type", $stop->stoptype);
        } else {
            $db->where("status_id", StatusId::PICKUP);
            $db->where("stop_type", StopType::DROP);
        }

        return $db->count_all_results();
    }

    /**
     * @param \stdClass $stop
     * @param \CI_DB_mysqli_driver $db
     * @return int
     */
    private static function countStatuses(\stdClass $stop, \CI_DB_mysqli_driver $db) : int
    {
        $db->from('tb_stop_status');
        $db->where("stop_id", $stop->id);
        $db->where("shipment_id", $stop->shift_id);

        return $db->count_all_results();
    }

    /**
     * @param array $bookingIds
     * @param \CI_DB_mysqli_driver $db
     * @return \stdClass[]
     */
    private static function getOrderDetails(array $bookingIds, \CI_DB_mysqli_driver $db) : array
    {
        $questionmarks = str_repeat("?,", count($bookingIds)-1) . "?";

        $result = $db->query(
            "
            SELECT
                   pickup_company,
                   delivery_company,
                   pickup_address1,
                   delivery_address1,
                   pickup_city,
                   delivery_city,
                   pickup_pincode,
                   delivery_pincode,
                   pickup_country,
                   delivery_country,
                   company_code,
                   order_id
            FROM tb_orders
            WHERE order_id IN (" . $questionmarks . ")", $bookingIds
        )->result();
        
        //log_message("error","Qryyyyyyyyyyyyyy:".$db->last_query());

        return $result ?? [];
    }

    /**
     * Get KN Tracking numbers for all bookings assigned for stop details
     *
     * @param \CI_DB_mysqli_driver $db Database driver
     * @param array  $bookingIds Array of booking Ids for all stop details
     * @param string $knTrackingNo SALOG KN Tracking number for a stop
     *
     * @return array
     */
    private static function getTruckingNumbers(
        \CI_DB_mysqli_driver $db,
        array $bookingIds,
        string $knTrackingNo
    ): array {
        if ((count($bookingIds) === 1) && $knTrackingNo != "") {
            return [$knTrackingNo];
        }

        $query = $db->query("
            SELECT
                (SELECT (CASE WHEN o.created_source = ? THEN (select ref_value from tb_order_references where order_id = o.id and reference_id = ? and status =? LIMIT 1) WHEN o.created_source = ? THEN (select ref_value from tb_order_references where order_id = o.id and reference_id = ? and status = ? LIMIT 1) ELSE '' END)) AS kn_tracking_no
            FROM tb_orders o WHERE o.order_id IN ?", [5, 'XSR', 1, 18, 'AWB', 1, $bookingIds]
        );
        $queryResult = $query->num_rows() > 0 ? $query->result_array() : [];
        foreach ($queryResult as $eachLine) {
            $response[] = $eachLine['kn_tracking_no'];
        }
        return $response ?? [];
    }
}
