<?php

namespace App\TripStop;

/**
 * class replaces gettripstopdetails function for individual shipments,
 * to work for multiple shipments, and cache&serve results
 */
class TripStopDetails
{
    private static ?object $instance = null;

    private object $cacheLayer;
    private array $shipments; // array of objects
    private array $results = [];
    private bool $rejected = false;

    /**
     * constructor is private for Singleton
     */
    private function __construct()
    {
        $this->cacheLayer = new TripStopDetailsModel();
    }

    /**
     * Singleton design pattern - getInstance
     *
     * @return TripStopDetails
     */
    public static function getInstance(): TripStopDetails {
        if (self::$instance === null) {
            self::$instance = new TripStopDetails();
        }

        return self::$instance;
    }

    /**
     * base setter for array of objects
     *
     * @param array $shipments
     */
    public function useShipments(array $shipments):void {
        $this->shipments = $shipments;
    }

    /**
     * specified parameter
     */
    public function useRejected(): void
    {
        $this->rejected = true;
    }

    /**
     * main processing function
     */
    public function executeQueries(): void
    {
        $this->cacheLayer->aggregateData($shipmentIds = array_map(
            static function($item) { return $item->id; },
            $this->shipments
        ));

        $this->processShipments();
    }

    /**
     * process multiple shipments at once
     */
    private function processShipments(): void
    {
        $this->results = [];
        foreach ($this->shipments as $shipment) {
            $this->results [$shipment->id] = $this->parseShipment($shipment);
        }
    }

    /**
     * previous gettripstopdetails() from new_helper.php
     *
     * @param object $shipment
     * @return object
     */
    private function parseShipment(object $shipment): object
    {
        $stopResults = $this->cacheLayer->getShiporderStops($shipment->id);

        $stopsData = [];
        $all_emps = [];
        $shipment->total_volume = 0;
        $shipment->SCAC = '';
        $shipment->stops = [];

        $stopDetailsDataset = $this->cacheLayer->getEmployeeStopOrDrop(array_keys($stopResults));

        foreach ($stopResults as $stop_id => $row1) {
            $row1["stopstatus"] = "0";
            $shipment_id = $shipment->id;
            $early_pickup = $late_pickup = $row1['startdate'];
            $early_delivery = $late_delivery = $row1['enddate'];

            $empsResults = array_filter([
                $stopDetailsDataset['stop'][$stop_id] ?? null,
                $stopDetailsDataset['drop'][$stop_id] ?? null,
            ]);
            $type = $emps = [];
            $row1["act_ship_units"] = 0;
            $accepted = $order_row_id = 0;

            foreach ($empsResults as $emp) {
                $emps[] = $emp->id;
                $order_id = $emp->order_id;
                if ($order_id != "") {
                    $this->updatePickupDates($order_id, $early_pickup, $late_pickup, $early_delivery, $late_delivery, $order_row_id);
                }

                $row1["container_no"] = $order_row_id
                    ? $this->fetchContainerNumbers($order_row_id, $shipment)
                    : "";

                if ($emp->accepted != "1") {
                    ++$accepted;
                }
                if (!in_array($emp->id, $all_emps)) {
                    $shipment->total_volume += $emp->shipment_volume;
                }
                $all_emps[] = $emp->id;
                if ($stop_id == $emp->stop_id) {
                    $type[] = "P";
                    $row1['startdate'] = $early_pickup;
                    $row1['enddate'] = $late_pickup;
                } else {
                    $type[] = "D";
                    $row1['startdate'] = $early_delivery;
                    $row1['enddate'] = $late_delivery;
                }
                $row1["act_ship_units"] += $emp->no_of_pkgs;
            }
            $type = array_unique($type);
            $row1["stoptype"] = "";
            if (count($type) === 2) {
                $row1["stoptype"] = "P/D";
            } elseif (count($type) === 1) {
                $row1["stoptype"] = $type[0];
                if ($row1['stoptype'] === 'P') {
                    $row1['startdate'] = $early_pickup;
                    $row1['enddate'] = $late_pickup;
                } else {
                    $row1['startdate'] = $early_delivery;
                    $row1['enddate'] = $late_delivery;
                }
            }

            $row1["stopstatus"] = "0";
            if (count($emps) > 0) {
                $row1["stopstatus"] = $this->fetchStopStatus($emps, $stop_id, $shipment_id, $accepted);
            }

            $stopsData[] = $row1;
        }

        $shipment->stops = $stopsData;

        return $shipment;
    }

    private function updatePickupDates($order_id, &$early_pickup, &$late_pickup, &$early_delivery, &$late_delivery, &$order_row_id): void
    {
        $getpickupdates = $this->cacheLayer->getPickupDates($order_id);

        if (empty($getpickupdates)) {
            return;
        }

        $early_pickup = $getpickupdates->early_pickup;
        $late_pickup = $getpickupdates->late_pickup;
        $early_delivery = $getpickupdates->early_delivery;
        $late_delivery = $getpickupdates->late_delivery;

        $order_row_id = $getpickupdates->id;
    }

    private function fetchContainerNumbers(int $order_row_id, object $shipment): string
    {

        if (!$order_row_id) {
            return "";
        }

        $getcontainer_no = $this->cacheLayer->getReferenceValues($order_row_id);
        if (empty($getcontainer_no)) {
            return "";
        }

        $containers = [];
        foreach ($getcontainer_no as $ctr) {
            if ($ctr->reference_id === 'CTR') {
                $containers[] = $ctr->ref_value;
            } elseif ($ctr->reference_id === 'SCAC') {
                $shipment->SCAC = $ctr->ref_value;
            }
        }

        return implode(',', $containers);
    }

    private function fetchStopStatus(array $emps, string $stop_id, string $shipment_id, int $accepted): int
    {
        [$statusOne, $statusOthers] = $this->cacheLayer->getStopStatus($emps, $stop_id, $shipment_id);

        if ($statusOne) {
            return $statusOne >= count($emps) ? 2 : 1;
        }

        if ($statusOthers > 0) {
            return 1;
        }
        if ($this->rejected && count($emps) === $accepted) {
            return 3;
        }

        return 0;
    }

    /**
     * getter for specified shipment id
     * from whole dataset
     *
     * @param string $shipmentId
     * @return object
     */
    public function getDetailsData(string $shipmentId): object
    {
        return $this->results[$shipmentId];
    }

}
