<?php

require 'vendor/nesbot/carbon/src/Carbon/Carbon.php';
require 'vendor/nesbot/carbon/src/Carbon/CarbonInterval.php';

use Carbon\Carbon;

/**
 * Application jwt helper
 */
if (!function_exists("generate_jwt")) {
    function generate_jwt(array $data, string $sourceType = 'driver'): string
    {
        $claim = $data;
        $claim["iat"] = strtotime("now");
        $claim["exp"] = $claim['exp'] ?? $claim["iat"] + 86400;
        $claim["iss"] = $claim['iss'] ?? "kn";
        $base64_header = base64Encode(json_encode(["typ" => "JWT", "alg" => "HS256"]));
        $base64_payload = base64Encode(json_encode($claim));
        $signature = hash_hmac('sha256', $base64_header . "." . $base64_payload, getSignatureKey($sourceType), true);
        $base64_signature = base64Encode($signature);
        return $base64_header . "." . $base64_payload . "." . $base64_signature;
    }
}

if (!function_exists("validate_jwt")) {
    function validate_jwt(string $jwt, string $sourceType = 'driver'): array
    {
        try {
            $tokenParts = explode('.', $jwt);
            if (2 == count($tokenParts)) {
                $tokenParts = [ROADLOG_JWT_HEADER, $tokenParts[0], $tokenParts[1]];
            }
            $header = base64_decode($tokenParts[0]);
            $payload = base64_decode($tokenParts[1]);
            $token_signature = str_replace(['/', '+', '=', '-', '_'], '', $tokenParts[2]);
            // check the expiration time - note this will cause an error if there is no 'exp' claim in the token
            // $expiration = Carbon::createFromTimestamp(json_decode($payload)->exp);
            $issuedTime = Carbon::createFromTimestamp(json_decode($payload)->iat);
            $expiresOn = Carbon::createFromTimestamp(strtotime($issuedTime . ' +' . JWT_TOKEN_VALID_DAYS . ' days'));
            $token_expired = (Carbon::now()->diffInSeconds($expiresOn, false) < 0);
            // header base64 encoded
            $base64_header = base64Encode($header);
            // payload base64 encoded
            $base64_payload = base64Encode($payload);
            // Create Signature Hash
            $signature = hash_hmac(
                'sha256',
                $base64_header . "." . $base64_payload,
                getSignatureKey($sourceType),
                true
            );
            // signature base64 encoded
            $base64_signature = base64Encode($signature);
            // verify it matches the signature provided in the token
            $signature_valid = ($base64_signature === $token_signature);
            if ($signature_valid) {
                if ($token_expired) {
                    return ['code' => 403, 'error' => 'expired', 'error_message' => 'token_expired'];
                } else {
                    return ['code' => 200, 'token' => json_decode($payload, true)];
                }
            } else {
                return ['code' => 400, 'error' => 'invalid', 'error_message' => 'signature_invalid'];
            }
        } catch (Exception $e) {
            return ['code' => $e->getCode(), 'error' => $e->getMessage(), 'error_message' => $e->getMessage()];
        }
    }
}

if (!function_exists("base64Encode")) {
    function base64Encode(string $text): string
    {
        return str_replace(['/', '+', '=', '-', '_'], '', base64_encode($text));
    }
}

if (!function_exists("generate_jwt_carrier")) {
    function generate_jwt_carrier(array $data): string
    {
        return generate_jwt($data, 'carrier');
    }
}

if (!function_exists("validate_jwt_carrier")) {
    function validate_jwt_carrier(string $jwt): array
    {
        return validate_jwt($jwt, 'carrier');
    }
}

if (!function_exists("getSignatureKey")) {
    function getSignatureKey(string $sourceType = ''): string
    {
        $sourceTypes = ['driver', 'carrier'];
        if (in_array($sourceType, $sourceTypes)) {
            return ('driver' === $sourceType) ? JWT_SIGNATURE : JWT_SIGNATURE_CARRIER;
        } else {
            throw new RuntimeException('Invalid source type is received');
        }
    }
}
?>
