<?php
class AtsService
{

    private $ci;
    private $company_code;
    private $branch_code;
    private $user_id;

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->model('common');
    }

    public function getUserId(): int
    {
        return $this->user_id;
    }

    public function getBranchCode(): string
    {
        return $this->branch_code;
    }

    public function getCompanyCode(): string
    {
        return $this->company_code;
    }

    public function ats_client(int $party_master_id, string $branch_code, string $company_code, int $user_id)
    {
        $this->company_code = $company_code;
        $this->branch_code = $branch_code;
        $this->user_id = $user_id;
        $where = ["id" => $party_master_id];
        $select = [
            'id',
            'name',
            'street',
            'location_id as city',
            'state',
            'country',
            'pincode as zip'
        ];
        $customerDetails = $this->ci->Common->gettblrowdata($where, $select, 'tbl_party_master', 0, 0);
        if (sizeof($customerDetails) > 0 && isset($customerDetails['id']) && is_numeric($customerDetails['id'])) {
            // generate xml format.
            $customer_address = $this->ats_xml_format($customerDetails);
        } else {
            return $this->send_response(1);
        }
        $res = $this->ats_soap_call($customer_address);
        if (false === $res) {
            return $this->send_response(2);
        }
        $xml = new SimpleXMLElement($res);
        if ($xml === false) {
            return $this->send_response(2);
        } else {
            $xml->registerXPathNamespace('c', '');
            $result = $xml->xpath('//ns1:checkPersonResponse');
            $responses = $result[0]->responses;
            $authentication = $responses->authentication;
            $response = $responses->response;
            $status_info = $response->statusinfo;
            if (isset($response->resultrow)) {
                $resultrows = $response->resultrow;
                if ((!empty($resultrows)) && sizeof($resultrows) > 0) {
                    $customer_info = array();
                    foreach ($resultrows as $resultrow) {
                        $customer_info[] = $resultrow;
                    }
                    return $customer_info;
                }
            } else {
                return $this->send_response(3);
            }
        }
    }

    public function ats_xml_format(array $customerDetails): string
    {
        $this->ci->load->library('AtsCustomer', $customerDetails);

        $company_code = $this->getCompanyCode();
        $branch_code = $this->getBranchCode();

        $customer_address = '';
        $customer_address = $this->getATSRequestPrefix();
        $customer_address .= '<requestrow xsi:type="zer:adressType">';
        $customer_address .= '<name xsi:type="xsd:string">' . $this->ci->atscustomer->getName() . '</name>';
        $customer_address .= '<custno xsi:type="xsd:string">' . $this->ci->atscustomer->getId() . '</custno>';
        $customer_address .= '<street xsi:type="xsd:string">' . $this->ci->atscustomer->getStreet() . '</street>';
        $customer_address .= '<city xsi:type="xsd:string">' . $this->ci->atscustomer->getCity() . '</city>';
        $customer_address .= '<zip xsi:type="xsd:string">' . $this->ci->atscustomer->getZip() . '</zip>';
        $customer_address .= '<country xsi:type="xsd:string">' . $this->ci->atscustomer->getCountry() . '</country>';
        $customer_address .= '<ftx xsi:type="xsd:string"></ftx>';
        $customer_address .= '<ftx3 xsi:type="xsd:string">' . $branch_code . '</ftx3>';
        $customer_address .= '<ftx4 xsi:type="xsd:string">' . $company_code . '</ftx4>';
        $customer_address .= '</requestrow>';
        $customer_address .= '</request>';
        $customer_address .= $this->getATSRequestSuffix();
        return $customer_address;
    }

    /*
     *   Purpose: To send error codes.
     */

    public function send_response($status_code): string
    {
        $response = '';
        switch ($status_code) {
            case 1:
                $response = "Please provide valid customer details";
                break;
            case 2:
                $response = "Invalid xml response";
                break;
            case 3:
                $response = "No issues, please continue next process.";
                break;
            default:
                break;
        }
        return $response;
    }

    /*
     * @param   party_master_id     numeric   AI field in tbl_party_master
     * @param   status_bi           String    only we allow following values 'H' || 'O' || 'C'
     * returns  json
     *
     *  Status bit values
     *      H for Hold, O for Override and  C for cancel
     */

    public function ats_status_update($party_master_id, $status_bit)
    {
        $response = '';
        /*
         *  Validation
         */
        $ats_status = array("H", "O", "C");
        if (!(is_numeric($party_master_id) && $party_master_id > 0)) {
            $response = array("status" => "0", "message" => "Please provide valid party master Id-");
            return $this->ats_status_update_response($response);
        }
        if (!in_array($status_bit, $ats_status)) {
            $response = array("status" => "0", "message" => "Please provide valid ats status codes");
            return $this->ats_status_update_response($response);
        }

        /*
         *  DB
         */
        $where = array('id' => $party_master_id);
        $party_master_info = $this->ci->common->gettblrowdata($where, 'id', 'tbl_party_master', 0, 0);
        if ((!empty($party_master_info)) && isset($party_master_info['id'])) {
            // update now.
            if ($status_bit == "C") {
                // no need to do any thing if status is "Cancel" OR "C"
                $response = array("status" => "1", "message" => "Ats Status updated successfully");
                return $this->ats_status_update_response($response);
            }
            $set = array("ats_status" => $status_bit);
            $this->ci->common->updatetbledata('tbl_party_master', $set, $where);
            // response
            $response = array("status" => "1", "message" => "Ats Status updated successfully");
            return $this->ats_status_update_response($response);
        } else {
            $response = array("status" => "0", "message" => "Please provide valid party master id.");
            return $this->ats_status_update_response($response);
        }
    }

    public function ats_status_update_response($response)
    {
        return json_encode($response);
    }

    public function get_ats_service()
    {
        return 'ats/ats_assets';
    }

    public function ats_soap_call(string $xml)
    {
        $response = false;
        $error_message = "";
        try {
            $remote_response = ats_soap_curl($xml);
            if (200 === $remote_response->getStatus()) {
                $response = $remote_response->getBody();
            } else {
                $error_message = $remote_response->getReasonPhrase() . '/' . $remote_response->getStatus();
            }
        } catch (Exception $e) {
            $error_message = $e->getMessage() . '/' . $e->getCode();
        }
        if (false === $response) {
            $this->saveToAtsLogs(['request' => $xml, 'response' => '', 'reason_for_failure' => $error_message]);
        }
        return $response;
    }

    /*
     *    Purpose : Based on address array we will verify over ats server.
     *    @param    ats_parites    array
     *    return    int
     */

    public function ats_client_bulk(array $ats_parties, string $branch_code, string $company_code, int $user_id)
    {
        $this->company_code = $company_code;
        $this->branch_code = $branch_code;
        $this->user_id = $user_id;
        if ((!empty($ats_parties)) && is_array($ats_parties) && sizeof($ats_parties) > 0) {
            // generate xml format.
            $customer_address = $this->ats_xml_format_bulk($ats_parties);
        } else {
            return $this->send_response(1);
        }
        $headers = array(
            "Content-Type: application/xml",
        );
        $res = $this->ats_soap_call($customer_address);
        if (false === $res) {
            return $this->send_response(2);
        }
        $xml = new SimpleXMLElement($res);
        if ($xml === false) {
            return $this->send_response(2);
        } else {
            $xml->registerXPathNamespace('c', '');
            $result = $xml->xpath('//ns1:checkPersonResponse');
            $responses = $result[0]->responses;
            $authentication = $responses->authentication;
            $response = $responses->response;
            $status_info = $response->statusinfo;
            if (isset($response->resultrow)) {
                $resultrows = $response->resultrow;
                if ((!empty($resultrows)) && sizeof($resultrows) > 0) {
                    $customer_info = array();
                    foreach ($resultrows as $resultrow) {
                        $customer_info[] = $resultrow;
                    }
                    /*
                     *   Generate Customer table
                     */
//                    return $this->generate_customer_table($customer_info, $party_master_id);
                    return $customer_info;
                }
            } else {
                return $this->send_response(3);
            }
        }
    }

    /*
     *    Purpose : Based on address array we will verify over ats server.
     *    @param    ats_parites    array
     *    return    int
     */

    public function ats_xml_format_bulk($ats_parties): string
    {
        $customer_address = '';
        $customer_address .= $this->getATSRequestPrefix();
        $customer_address .= $this->generate_multiple_request_rows($ats_parties);
        $customer_address .= '</request>';
        $customer_address .= $this->getATSRequestSuffix();
        return $customer_address;
    }

    /*
     *  Purpose : To generate multiple requestrows for array of party address
     *  @param      atr_parties     array
     *  return      xml string
     */

    public function generate_multiple_request_rows($ats_parties): string
    {
        $customer_address = '';
        $i = 0;
        $company_code = $this->getCompanyCode();
        $branch_code = $this->getBranchCode();

        foreach ($ats_parties as $party_info) {
            $name = $party_info['name'] ?? '';
            $city = $party_info['city'] ?? '';
            $zipcode = $party_info['zipcode'] ?? '';
            $country = $party_info['country'] ?? '';
            $street = $party_info['street'] ?? '';
            $booking_id = $party_info['booking_id'] ?? '';
            $type = $party_info['type'] ?? ''; // shipper,consignee,
            $customer_address .= '<requestrow xsi:type="zer:adressType">';
            $customer_address .= '<name xsi:type="xsd:string">' . $name . '</name>';
            $customer_address .= '<custno xsi:type="xsd:string">' . ++$i . '</custno>';
            $customer_address .= '<street xsi:type="xsd:string">' . $street . '</street>';
            $customer_address .= '<city xsi:type="xsd:string">' . $city . '</city>';
            $customer_address .= '<zip xsi:type="xsd:string">' . $zipcode . '</zip>';
            $customer_address .= '<country xsi:type="xsd:string">' . $country . '</country>';
            $customer_address .= '<ftx xsi:type="xsd:string"></ftx>';
            $customer_address .= '<ftx2 xsi:type="xsd:string">' . $booking_id . '</ftx2>';
            $customer_address .= '<ftx3 xsi:type="xsd:string">' . $branch_code . '</ftx3>';
            $customer_address .= '<ftx4 xsi:type="xsd:string">' . $company_code . '</ftx4>';
            $customer_address .= '<ftx5 xsi:type="xsd:string">' . $type . '</ftx5>';
            $customer_address .= '</requestrow>';
        }

        return $customer_address;
    }

    public function saveToAtsLogs($insertData): void
    {
        $insertData['user_id'] = $this->getUserId();
        $insertData['company_code'] = $this->getCompanyCode();
        $insertData['branch_code'] = $this->getBranchCode();
        $this->ci->common->insertTableData('ats_logs', $insertData);
    }

    public function updateToAtsLogs(array $updatedInfo, int $id): void
    {
        $this->ci->common->updatetbledata('ats_logs', $updatedInfo, ['id' => $id]);
    }

    public function atsLogsHandler(array $response, int $id): void
    {
        $this->updateToAtsLogs($response, $id);
    }

    public function getATSRequestPrefix(): string
    {
        $atsPrefix = '';
        $atsPrefix .= '<?xml version="1.0" encoding="UTF-8"?>';
        $atsPrefix .= '<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:bin="http://www.diehl-informatik.de/Zerberus/binding" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">';
        $atsPrefix .= '<soapenv:Header />';
        $atsPrefix .= '<soapenv:Body>';
        $atsPrefix .= '<bin:checkPerson soapenv:encodingStyle="http://schemas.xmlsoap.org/soap/encoding/">';
        $atsPrefix .= '<request xmlns:zer="http://www.diehl-informatik.de/Zerberus.xsd1" xsi:type="zer:requestType">';
        $atsPrefix .= '<useBadGuy xsi:type="xsd:boolean">true</useBadGuy>';
        $atsPrefix .= '<useGoodLink xsi:type="xsd:boolean">true</useGoodLink>';
        $atsPrefix .= '<authentication xsi:type="zer:authenticationType">';
        $atsPrefix .= '<version xsi:type="xsd:string">' . ATS_VERSION . '</version>';
        $atsPrefix .= '<customernr xsi:type="xsd:string">' . ATS_CUSTOMERNR . '</customernr>';
        $atsPrefix .= '<reference xsi:type="xsd:string">' . ATS_REFERENCE . '</reference>';
        $atsPrefix .= '</authentication>';
        return $atsPrefix;
    }

    public function getATSRequestSuffix(): string
    {
        return '</bin:checkPerson></soapenv:Body></soapenv:Envelope>';
    }
}
