<?php

require_once './vendor/pear/http_request2/HTTP/Request2.php';


class Carrierweightupdate
{
    private $ci;
    private const DELHIVERY_ID = "310";
    private const ECOM_EXPRESS_ID = "311";
    private const GRAMS = "G";
    private const GRAMS_ARRAY = ["G", "Gms", "gms", "grm"];
    private const KILOGRAMS_ARRAY = ["Kg", "kg"];
    private const TONS_ARRAY = ["Tons", "tons"];
    private const HTTP_STATUS_200 = 200;

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->model('common');
        $this->ci->load->helper('log_helper');
    }

    /**
     * @param $orderId
     * @return void
     */
    public function updateCarrierWeight($orderId): void
    {
        if ($orderId !== null) {
            $orderWeights = $this->getWeightsForOrder($orderId);

            $this->updateOrderWithFoundWeights($orderWeights, $orderId);
        }
    }

    /**
     * @param int $orderId
     * @return array
     */
    private function getWeightsForOrder(int $orderId): array
    {
        $this->ci->db->select('cd.weight, cd.weight_unit');
        $this->ci->db->from('tb_cargo_details as cd');
        $this->ci->db->join('tb_order_cargodetails as ocd', 'ocd.cargo_id = cd.id', 'LEFT');
        $this->ci->db->where('ocd.order_id', $orderId);

        return $this->ci->db->get()->result_array();
    }

    /**
     * @param array $orderWeights
     * @param int $orderId
     * @return void
     */
    private function updateOrderWithFoundWeights(array $orderWeights, int $orderId): void
    {
        if (!empty($orderWeights)) {
            $sumWeight = $this->convertWeightToGramsAndSum($orderWeights);
            $vendorId = $this->findVendorId($orderId);
            $docketNumber = $this->checkDocketNumber($orderId);
            $dataToUpdate = $this->dataToUpdate($docketNumber, $sumWeight, $orderId);

            $this->chooseCarrierAndUpdateOrder($vendorId, $dataToUpdate);
        }
    }

    /**
     * @param array $orderWeights
     * @return int
     */
    private function convertWeightToGramsAndSum(array $orderWeights): int
    {
        $sumWeight = 0;
        foreach ($orderWeights as $weight) {
            if (in_array($weight['weight_unit'], self::GRAMS_ARRAY)) {
                $sumWeight += $weight['weight'];
            } elseif (in_array($weight['weight_unit'], self::KILOGRAMS_ARRAY)) {
                $sumWeight += $weight['weight'] * 1000;
            } elseif (in_array($weight['weight_unit'], self::TONS_ARRAY)) {
                $sumWeight += $weight['weight'] * 1000000;
            }
        }

        return $sumWeight;
    }

    /**
     * @param int $orderId
     * @return int
     */
    private function findVendorId(int $orderId): int
    {
        $this->ci->db->select('vendor_id');
        $this->ci->db->from('tb_orders');
        $this->ci->db->where('id', $orderId);
        $vendor = $this->ci->db->get()->result_array();

        return $vendor[0]['vendor_id'] ?? 0;
    }

    /**
     * @param string $docketNumber
     * @param int $sumWeight
     * @param int $orderId
     * @return array
     */
    private function dataToUpdate(string $docketNumber, int $sumWeight, int $orderId): array
    {
        $response['docket_number'] = $docketNumber;
        $response['ordgross_weight'] = $sumWeight;
        $response['order_id'] = $orderId;
        $response['weight_unit'] = self::GRAMS;

        return $response;
    }

    /**
     * @param int $orderId
     * @return string
     */
    public function checkDocketNumber(int $orderId): string
    {
        $response = "0";
        $where = ["order_id" => $orderId, "reference_id" => "AWB", "ref_value <>" => "", "status" => 1];
        $orderReferences = $this->ci->common->gettblrowdata($where, "ref_value", "tb_order_references", 0, 0);

        if (count($orderReferences) > 0) {
            $response = $orderReferences['ref_value'];
        }

        log_error(
            UNIQLO_LOG_PREFIX . "CarrierWeightUpdate checkDocketNumber AWB exists for order: " . $orderId . ", AWB:" . $response
        );

        return $response;
    }

    /**
     * @param int $vendorId
     * @param array $dataToUpdate
     * @return void
     */
    private function chooseCarrierAndUpdateOrder(int $vendorId, array $dataToUpdate): void
    {
        if ($vendorId == self::DELHIVERY_ID) {
            $this->delhiveryOrderUpdate($dataToUpdate);
        } elseif ($vendorId == self::ECOM_EXPRESS_ID) {
            $this->ecomOrderUpdate($dataToUpdate);
        }
    }

    public function delhiveryOrderUpdate(array $orderData)
    {
        log_error(
            UNIQLO_LOG_PREFIX . "CarrierWeightUpdate delhiveryOrderUpdate inbound data: " . json_encode($orderData)
        );

        $xmlPostString = "Failed";

        if (!empty($orderData)) {
            $shipData = [
                "waybill" => (string)$orderData['docket_number'],
                "gm" => (float)$orderData['ordgross_weight']
            ];
            $requestData = json_encode($shipData);
            $ediResponse = "Fail";
            $status = 0;
            $params = [
                'edi_type' => 1, /*'1->EDI Transport Order,2->EDI Status'*/
                'transaction_id' => time(),
                'edi_id' => 8, /*edi id based on name take it from tb_edi_types*/
                'edi_name' => "Delhivery",
                'bounded_type' => 2, /*'1->Inbound(partner => kN),2->Outbound (KN => partner)'*/
                'edi_format_type' => 'json',
                'status' => $status,
                'obj_type' => 1,
                'txn_obj_id' => $orderData['order_id'],
                'user_id' => 244,
                'company_code' => "INKN",
                'branch_code' => "INCL",
                'edi_request' => $requestData,
                'edi_response' => $ediResponse
            ];
            $ediTransactionId = $this->ci->common->insertTableData('tb_etn_edi_transactions', $params);

            $request = new HTTP_Request2();
            $request->setUrl(DELHIVERY_ORD_EDIT);
            $request->setMethod(HTTP_Request2::METHOD_POST);

            $config = [
                'follow_redirects' => true,
                'ssl_verify_peer' => false,
                'connect_timeout' => 100,
                'timeout' => 120
            ];

            if (ENVIRONMENT !== "development") {
                $config['proxy'] = PROXY_CONSTANT_HERE;
            }

            $request->setConfig($config);
            $request->setHeader([
                                    'Authorization' => "Token " . DELHIVERY_AUTH_TOKEN,
                                    'Content-Type' => 'application/json'
                                ]);
            $request->setBody($requestData);

            try {
                $response = $request->send();

                if ($response->getStatus() == self::HTTP_STATUS_200) {
                    $responseBody = $response->getBody();

                    if ($this->isValidXml($responseBody) == true) {
                        $xml = simplexml_load_string($responseBody, "SimpleXMLElement", LIBXML_NOCDATA);
                        $json = json_encode($xml);
                        $decodedRespons = json_decode($json, true);
                    } else {
                        $decodedRespons = json_decode($responseBody, true);
                    }

                    $ediResponse = json_encode($decodedRespons);
                    log_error(
                        UNIQLO_LOG_PREFIX . 'CarrierWeightUpdate delhiveryOrderUpdate API response payload: ' . $ediResponse
                    );

                    if ($decodedRespons['status']) {
                        $xmlPostString = $decodedRespons['error'] ?? "";

                        if ($decodedRespons['status'] != "Failure") {
                            $xmlPostString = "Success";
                            $status = 1;
                        }
                    }
                } else {
                    $xmlPostString = 'Unexpected HTTP status: ' . $response->getStatus(
                        ) . ' ' . $response->getReasonPhrase();
                }
            } catch (HTTP_Request2_Exception $e) {
                $xmlPostString = 'Error: ' . $e->getMessage();
            }

            /* here getting success and return awb number and eTN order number*/
            $set = ["status" => $status, "edi_response" => $ediResponse];

            $this->ci->common->updatetbledata("tb_etn_edi_transactions", $set, ["id" => $ediTransactionId]);
        }

        return $xmlPostString;
    }

    /**
     * @param $content
     * @return bool
     */
    private function isValidXml($content): bool
    {
        $content = trim($content);

        if (empty($content) || stripos($content, '<!DOCTYPE html>') !== false) {
            return false;
        }

        libxml_use_internal_errors(true);
        simplexml_load_string($content);
        $errors = libxml_get_errors();
        libxml_clear_errors();

        return empty($errors);
    }

    /**
     * @param array $orderData
     * @return mixed|string
     */
    public function ecomOrderUpdate(array $orderData)
    {
        log_error(UNIQLO_LOG_PREFIX . "CarrierWeightUpdate ecomOrderUpdate inbound data: " . json_encode($orderData));

        $responseString = "Failed";

        if (!empty($orderData)) {
            $orderData['ordgross_weight'] = (float)$orderData['ordgross_weight'];
            $weightUnit = $orderData['weight_unit'] ?? "Gms";

            if ($orderData['ordgross_weight'] > 0 && in_array($weightUnit, self::GRAMS_ARRAY)) {
                $orderData['ordgross_weight'] = round(($orderData['ordgross_weight'] / 1000), 2);
            }

            if ($orderData['ordgross_weight'] <= 0) {
                $orderData['ordgross_weight'] = 0.1;
            }

            $shipData[] = [
                "AWB" => $orderData['docket_number'],
                "Length" => 0,
                "Width" => 0,
                "Height" => 0,
                "Weight" => $orderData['ordgross_weight']
            ];
            $requestData = json_encode($shipData);
            $ediResponse = "Fail";
            $status = 0;
            $params = [
                'edi_type' => 1, /* 1->EDI Transport Order, 2->EDI Status */
                'transaction_id' => time(),
                'edi_id' => 9, /* edi id based on name take it from tb_edi_types */
                'edi_name' => "Ecom Express",
                'bounded_type' => 2, /*'1->Inbound(partner => kN),2->Outbound (KN => partner)'*/
                'edi_format_type' => 'json',
                'status' => $status,
                'obj_type' => 1,
                'txn_obj_id' => $orderData['order_id'],
                'user_id' => 244,
                'company_code' => "INKN",
                'branch_code' => "INCL",
                'edi_request' => $requestData,
                'edi_response' => $ediResponse
            ];

            $ediTransactionId = $this->ci->common->insertTableData('tb_etn_edi_transactions', $params);
            $request = 'username=' . ECOM_EXP_UNAME . '&password=' . ECOM_EXP_PWD . '&json_input=' . $requestData;
            $proxy = defined('VZEN_PROXY') ? VZEN_PROXY : '';
            $ecomResponse = curlRequestor(
                ECOM_EXP_EDIT_API,
                'POST',
                $request,
                $proxy,
                [],
                retryDecider(),
                retryDelay()
            );

            $responseBody = json_decode($ecomResponse['body'], true);

            log_error(
                UNIQLO_LOG_PREFIX . 'CarrierWeightUpdate ecomOrderUpdate API response payload: ' . json_encode(
                    $responseBody
                )
            );

            log_error(
                UNIQLO_LOG_PREFIX . 'CarrierWeightUpdate ecomOrderUpdate EcomExpress order API request for data ' . $requestData .
                ($ecomResponse['respCode'] === self::HTTP_STATUS_200 ? ' succeeded.' : ' failed with: ' . $ecomResponse['curlError'])
            );

            if (!empty($responseBody)) {
                $ediResponse = json_encode($responseBody);

                if (isset($responseBody['shipments'])) {
                    $response = $responseBody['shipments'][0] ?? [];
                    $responseString = $response['reason'] ?? "";

                    if (isset($response['success']) && $response['success']) {
                        $status = 1;
                        $responseString = "Success";
                    }
                }
            }

            /* here getting success and return awb number and eTN order number*/
            $set = ["status" => $status, "edi_response" => $ediResponse];

            $this->ci->common->updatetbledata("tb_etn_edi_transactions", $set, ["id" => $ediTransactionId]);
        }

        return $responseString;
    }
}