<?php

use phpseclib\Net\SFTP;

require 'vendor/autoload.php';

class D3_dms
{
    const PDF_TEMPORARY_DIRECTORY = FCPATH .'assets/tmp';

    private $ci;

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->helper('log');
        $this->ci->load->library('DocumentSequenceLibrary');
    }

    public function generate_jpl(array $request): string
    {
        switch (true) {
            case !isset($request['order_id']):
                self::throwException('Order ID is undefined');

            case !isset($request['branch_code']):
                self::throwException('Branch code is undefined for order \'' . $request['order_id'] . '\'');

            case !isset($request['assoc_id']):
                self::throwException('Voyage number is undefined for order \'' . $request['order_id'] . '\'');

            case !isset($request['upload_type']):
                self::throwException('Upload type is undefined for order \'' . $request['order_id'] . '\'');

            case !isset($request['file_path']):
                self::throwException('File path is undefined for order \'' . $request['order_id'] . '\'');

            case !isset($request['ref_value_kn_office']):
                self::throwException('KN office reference is undefined for order \'' . $request['order_id'] . '\'');

            case !isset($request['physical_sender']):
                self::throwException('Physical sender is undefined for order \'' . $request['order_id'] . '\'');

            case @mkdir(D3DMS_TEMP_PATH, 0700, true) === false && !is_dir(D3DMS_TEMP_PATH):
                self::throwException('Unable to create directory \'' . D3DMS_TEMP_PATH . '\'');
        }

        $documentPath = $request['file_path'];
        $documentExtension = strtolower(pathinfo($documentPath, PATHINFO_EXTENSION));
        $countryCode = substr($request['branch_code'], 0, 2);
        $pictureType = $request['upload_type'];
        $documentId = $this->getDocumentId($countryCode, $pictureType);
        $localDocumentFilePath = D3DMS_TEMP_PATH . '/' . $documentId . '.pdf';
        $pdf = new \Mpdf\Mpdf(['tempDir' => self::PDF_TEMPORARY_DIRECTORY]);
        $pdf->showImageErrors = true;
        if ($documentExtension != "pdf") {
            try {
                $pages = $pdf->SetSourceFile($documentPath);

                for ($page = 1; $page <= $pages; $page++) {
                    $pdf->useTemplate($pdf->importPage($page));

                    if ($page < $pages) {
                        $pdf->addPage();
                    }
                }
            } catch (\Exception $exception) {
                [$imageWidth, $imageHeight] = @getimagesize($documentPath);

                if (!$imageWidth || !$imageHeight) {
                    self::throwException('Unable to convert document for order \'' . $request['order_id'] . '\' to PDF because its type is invalid');
                }

                try {
                    $pdf->writeHtml('');
                    $pdf->Image($documentPath, 5, 5, $imageWidth, $imageHeight, self::getFileExtension($documentPath), '', true, true);
                } catch (\Exception $exception) {
                    self::throwException('Unable to convert document to PDF for order ' . $request['order_id'] . ':' . $exception->getMessage());
                }
            }
            try {
                $pdf->Output($localDocumentFilePath);
            } catch (\Exception $exception) {
                self::throwException('Unable to generate PDF in \'' . D3DMS_TEMP_PATH . '\' for order \'' . $request['order_id'] . '\'');
            }
        } else {
            try {
                copy($documentPath, $localDocumentFilePath);
            } catch (\Exception $exception) {
                self::throwException('Unable to copy PDF to \'' . D3DMS_TEMP_PATH . '\'');
            }
        }

        $localJplFilePath = D3DMS_TEMP_PATH . '/' . $documentId . '.jpl';

        $now = time();
        $archivingDate = date('d.m.Y', $now);
        $knOfficeRereference = $request['ref_value_kn_office'];

        $jplContents = [
            'dokuart = "LWIS"',
            'dok_dat_feld[2] = "' . $countryCode . '"',
            'dok_dat_feld[3] = "' . substr($request['branch_code'], 2) . '"',
            'dok_dat_feld[7] = "' . $request['order_id'] . '"',
            'dok_dat_feld[11] = "' . $now . '"',
            'dok_dat_feld[12] = "SVKONEKT"',
            'dok_dat_feld[13] = "' . $knOfficeRereference . '"',
            'dok_dat_feld[43] = "' . $documentId . '"',
            'dok_dat_feld[44] = "' . self::getJplPictureType($pictureType) . '"',
            'dok_dat_feld[50] = "' . $archivingDate . '"',
            'dok_dat_feld[81] = "' . $pdf->SetSourceFile($localDocumentFilePath) . '"',
            'dok_dat_feld_60[1] = "COMREF"',
            'dok_dat_feld_61[1] = "' . $knOfficeRereference . '"',
            '#senderId="MACADAM"',
            '#senderInst="' . self::getEnvironment($request['physical_sender']) . '"'
        ];

        if (@file_put_contents($localJplFilePath, join(PHP_EOL, $jplContents)) === false) {
            self::throwException('Unable to write in JPL file \'' . $localJplFilePath . '\' for order \'' . $request['order_id'] . '\'');
        }

        self::uploadToSftp($localJplFilePath, $localDocumentFilePath);

        return $documentId;
    }

    private static function getPictureId(string $pictureType): int
    {
        switch ($pictureType) {
            case 'POD':
                return 1;

            case 'POP':
                return 2;

            default:
                return 3;
        }
    }

    private function getDocumentId(string $countryCode, string $pictureType): string
    {
        return $this->ci->documentsequencelibrary->getSequence('ETN', $countryCode, self::getPictureId($pictureType));
    }

    private static function getJplPictureType(string $pictureType): int
    {
        switch ($pictureType) {
            case 'POD':
                return 545;

            case 'POP':
                return 507;

            default:
                return 999;
        }
    }

    private static function uploadToSftp(string...$files): void
    {
        $sftp = new SFTP(SWIFLOG_FTP);

        if (!$sftp->login(SWIFLOG_FTP_USER, SWIFLOG_FTP_PWD)) {
            unset($sftp);
            self::throwException('Unable to login in SFTP server \'' . SWIFLOG_FTP . '\' with user \'' . SWIFLOG_FTP_USER . '\': ');
        }

        if (!$sftp->chdir(D3DMS_OUTBOUND)) {
            self::throwException('Unable to go to directory \'' . D3DMS_OUTBOUND . '\' in SFTP server \'' . SWIFLOG_FTP . '\' with user \'' . SWIFLOG_FTP_USER . '\': ');
        }

        foreach ($files as $file) {
            if (!$sftp->put(basename($file), $file, SFTP::SOURCE_LOCAL_FILE)) {
                self::throwException('Unable to upload \'' . $file . '\' to directory \'' . D3DMS_OUTBOUND . '\' in SFTP server \'' . SWIFLOG_FTP . '\' with user \'' . SWIFLOG_FTP_USER . '\': ');
            }

            // @TODO Set up an asynchronious garbage collector to delete old files not deleted when PUT failed
            if (unlink($file) === false) {
                self::throwException('Unable to delete \'' . $file . '\' from file system');
            }
        }
        unset($sftp);
    }

    private static function getEnvironment(string $physicalSender): string
    {
        // @TODO define a configuration directive and use it here instead of hardcode legit physicals sender here.
        switch ($physicalSender) {
            case 'KNALP01':
                return 'PROD';

            case 'KNALT03':
                return 'TEST_UAT';

            case 'KNALT01':
                return 'TEST';

            default:
                log_debug('Unable to define environement for physical sender \'' . $physicalSender . '\', use TEST');
                return 'TEST';
        }
    }

    private static function getFileExtension(string $path): string
    {
        $fileExtension = (new SplFileInfo($path))->getExtension();

        if (!$fileExtension) {
            self::throwException('Unable to get file extension from path \'' . $path . '\'');
        }

        return strtolower($fileExtension);
    }

    private static function throwException(string $message): void
    {
        log_error($message, 2);

        throw new \RuntimeException($message);
    }

}
