<?php

use App\TripStop\TripStopDetails;

final class Livetrackservice
{

    private object $CI;
    private array $userdata = [];

    public function __construct()
    {
        $this->CI = &get_instance();
        $this->CI->load->model('livetrackmodel');
    }

    public function setUserData(array $userdata): void
    {
        $this->userdata = [
            'userid' => $userdata['userid'] ?? $this->CI->session->userdata('user_id'),
            'curtz' => $userdata['curtz'] ?? $this->CI->session->userdata('usr_tzone')
        ];
    }

    public function getLiveCabDetailsData(): array
    {
        if (empty($this->userdata)) {
            $this->setUserData([]);
        }

        $dataset = $this->getVehiclesOfActiveTrips();
        $speedUpdates = $this->getSpeedUpdates($dataset);
        $locationNameUpdates = $this->getLocationNameUpdates($speedUpdates);

        foreach ($dataset as &$row) {
            $key = $row['vehicle_id'] ?? '-';

            $row['speed'] = $speedUpdates[$key]['speed'] ?? $row['speed'] ?? 0;
            $row['speed_badge_color'] = $speedUpdates[$key]['speed_badge_color'] ?? $row['speed_badge_color'] ?? 'green';
            $row['speed_multiplied'] = $speedUpdates[$key]['speed_multiplied'] ?? $row['speed_multiplied'] ?? 0;
            $row['latitude'] = $speedUpdates[$key]['latitude'] ?? $row['latitude'] ?? 0;
            $row['longitude'] = $speedUpdates[$key]['longitude'] ?? $row['longitude'] ?? 0;
            $row['location_name'] = $locationNameUpdates[$key]['location_name'] ?? '?*?*?';

            $row = (object)$row;
        }
        unset($row);

        usort($dataset, static function($a, $b) {
            if ($a->status === $b->status) {
                return strcmp($a->register_number, $b->register_number);
            }
            return $a->status - $b->status;
        });

        return $dataset;
    }

    public function getVshiftsData(): array
    {
        $data = $this->CI->livetrackmodel->fetchVehiclesShifts(
            $this->userdata['userid'],
            $this->userdata['curtz']
        );

        $state = TripStopDetails::getInstance();
        $state->useShipments($data);
        $state->useRejected();
        $state->executeQueries();

        foreach ($data as &$row) {
            $row->details = $state->getDetailsData($row->id);
        }

        return $data;
    }

    private function getVehiclesOfActiveTrips(): array
    {
        $userid = $this->userdata['userid'];
        $vehicles = $this->CI->livetrackmodel->fetchUserActiveVehicles($userid);
        $results = [];

        foreach ($vehicles as $row) {
            $results[$row->vehicle_id] = [
                'vehicle_id' => $row->vehicle_id,
                'register_number' => $row->register_number,
                'status' => 0,
                'tripid' => 0
            ];
        }

        $vehiclesOfActiveTrips = $this->CI->livetrackmodel->fetchVehiclesOfActiveTrips(array_keys($results));

        foreach ($vehiclesOfActiveTrips as $row) {
            $results [$row['vehicle_id']] = array_merge($results [$row['vehicle_id']] ?? [], $row);
        }

        return $results;
    }

    private function getSpeedUpdates(array $dataset): array
    {
        $vehicleIds = array_unique(array_column($dataset, 'vehicle_id'));
        $data = $this->CI->livetrackmodel->fetchSpeedUpdates($vehicleIds);

        $result = [];
        foreach ($data as $row) {
            [$speed_multiplied, $speed_badge_color] = $this->calculateDisplayedSpeed($row->speed ?: 0);

            $result [$row->id] = [
                'vehicle_id' => $row->id,
                'speed' => $row->speed ?: 0,
                'speed_multiplied' => $speed_multiplied,
                'speed_badge_color' => $speed_badge_color,
                'latitude' => $row->latitude,
                'longitude' => $row->longitude
            ];
        }

        return $result;
    }

    private function calculateDisplayedSpeed(float $speed): array
    {
        // this multiplier has not documented origin
        $speed_multiplied = 3.6 * $speed;

        $speed_badge_color = $speed_multiplied > 30 ? 'yellow' : 'green';
        if ($speed_multiplied >= 60) {
            $speed_badge_color = 'red';
        }

        return [$speed_multiplied, $speed_badge_color];
    }

    private function getLocationNameUpdates(array $dataset): array
    {
        $locations = [];
        foreach ($dataset as &$row) {
            $key = $row['latitude'] . ':' . $row['longitude'];
            $locations [$key] = null;
            $row ['location_key'] = $key;
            $row ['location_name'] = '';
        }
        unset ($row);

        [$foundLocations, $missingLocations] = $this->CI->livetrackmodel->fetchLocationNameForMultipleLocations(
            array_keys($locations)
        );

        $fixedMissingLocations = $this->whenLocationNameIsNotInDB(array_values($missingLocations));

        foreach ($dataset as &$row) {
            $row ['location_name'] = $foundLocations[$row['location_key']]
                ?? $fixedMissingLocations[$row['location_key']]
                ?? '';
        }

        return $dataset;
    }

    private function whenLocationNameIsNotInDB(array $locations): array
    {
        $results = [];
        $isChina = $this->CI->session->userdata('company_code') === "CNKN";

        foreach ($locations as $line) {
            if (!$line || trim($line) === ':') {
                continue;
            }
            [$lat, $lng] = explode(':', $line);

            if ($isChina) {
                $results [$line] = getLocationNameByHereMap($lat, $lng);
            } else {
                $results [$line] = getLocationNameByGoogleMap($lat, $lng);
            }
        }
        return $results;
    }

}
