<?php

use phpseclib\Net\SFTP;
use App\Libraries\OrderCreationSourceHelper;


class OpenboxMessageHandler
{
    private $ci;
    const ACTION_CREATE = 'CREATE';
    const ACTION_UPDATE = 'UPDATE';
    const ACCEPTED_STATUS_CODE = '0212';

    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->helper(['edivalidation', 'customerconfigedi', 'user', 'Log']);
        $this->ci->load->model(['common', 'OpenboxModel']);
    }

    public function pullOrderXml(): void
    {
        try {
            $sftp = new SFTP(OPENBOX_SERVER_IP);
            if (!$sftp->login(OPENBOX_USERNAME, OPENBOX_PASSWORD)) {
                unset($sftp);
                log_error('Open box-info: FTP Unable to Login');
                return;
            }

            if (!$sftp->chdir(OPENBOX_ORDERS_CREATE_PATH)) {
                log_error('Open box-info: Unable to find folder in FTP');
                return;
            }

            $files_list = $sftp->nlist();
            if (empty($files_list)) {
                log_error('Open box-info: Unable to find files in ftp.');
                return;
            }

            foreach ($files_list as $index => $filename) {
                if (strlen($filename) > 10) {
                    $file_content = $sftp->get(OPENBOX_ORDERS_CREATE_PATH . '/' . $filename);
                    /* Validate XML format */
                    if (validateXml($file_content)) {
                        $response = json_decode(json_encode(simplexml_load_string($file_content)), true);
                        /* Content validation  */
                        if ($this->validateXMLData($response) == true) {

                            $filterData = $this->orderData($response);

                            if (empty($filterData)) {
                                $sftp->put(OPENBOX_ORDERS_CREATE_PATH . '/' . 'Failed' . '/' . $filename, $file_content);
                                $sftp->delete(OPENBOX_ORDERS_CREATE_PATH . '/' . $filename);
                                continue;
                            }
                            if ($filterData['Action'] == self::ACTION_CREATE) {

                                if ($this->ci->OpenboxModel->saveOrder($filterData)) {

                                    $sftp->delete(OPENBOX_ORDERS_CREATE_PATH . '/' . $filename);
                                }
                            }
                            if ($filterData['Action'] == self::ACTION_UPDATE) {

                                if ($this->ci->OpenboxModel->updateOrder($filterData)) {
                                    $sftp->delete(OPENBOX_ORDERS_CREATE_PATH . '/' . $filename);
                                }
                            }
                        }
                    }
                }
            }
            unset($sftp);
        } catch (Exception $e) {
            log_error('Edi info: FTP error - ' . $e->getMessage());
        }
    }

    private function validateXMLData(array $data): bool
    {
        if (is_array($data['Action']) || $data['Action'] <> self::ACTION_CREATE && $data['Action'] <> self::ACTION_UPDATE) {
            log_error('OpenBox Order Create : Invalid Action.');
            return false;
        }

        if (!isset($data['Header']['DestinationApp']) || $data['Header']['DestinationApp'] !== 'SVKONEKT') {
            log_error('OpenBox Order Create : Destination App is invalid.');
            return false;
        }

        if (is_array($data['Header']['SequenceNumber']) || !isset($data['Header']['SequenceNumber']) || empty($data['Header']['SequenceNumber']) ||
            strlen($data['Header']['SequenceNumber']) > 10 || !is_numeric($data['Header']['SequenceNumber'])) {
            log_error('OpenBox Order Create : Sequence Number is invalid.');
            return false;
        }

        if (!isset($data['OrderDetails']['EXTOrderID']) ||
            strlen($data['OrderDetails']['EXTOrderID']) > 60
        ) {
            log_error('Open-box-info: External order id length exceeded.');
            return false;
        }

        if ($data['Action'] == self::ACTION_UPDATE) {
            $res = $this->ci->OpenboxModel->checkSequenceNumber(
                $data['OrderDetails']['EXTOrderID']
            );
            if ($res == 0 || $res >= $data['Header']['SequenceNumber']) {
                log_error('Open-box-info: Sequence Number should greater than current value.');
                return false;
            }

            $orderId = $this->ci->OpenboxModel->getOrderId(
                $data['OrderDetails']['EXTOrderID']
            );

            $checkAcceptStatus = $this->ci->OpenboxModel->getCount(
                'tb_stop_status',
                ['status_code' => self::ACCEPTED_STATUS_CODE, 'order_id' => $orderId, 'status' => 1]
            );

            if ($checkAcceptStatus != 0 ) {
                log_error('OpenBox Order Create :  Trip accepted by driver cannot update order now.'.$checkAcceptStatus);
                return false;
            }

        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['PlannedDateTime']['From'], $data['OrderDetails']['OrderStartLocation']['PlannedDateTime']['To'])) {
            log_error('OpenBox Order Create : Order start location date and time not set.');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['PlannedDateTime']['From'], $data['OrderDetails']['OrderEndLocation']['PlannedDateTime']['To'])) {
            log_error('OpenBox Order Create : Order end location date and time not set.');
            return false;
        }

        if (!isset($data['Header']['DestinationApp']) ||
            empty($data['Header']['DestinationApp']) ||
            $data['Header']['DestinationApp'] != 'SVKONEKT'
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Destination App');
            return false;
        }

        if (!isset($data['KNOrgDetails']['PhysicalReceiver']) ||
            is_array($data['KNOrgDetails']['PhysicalReceiver']) ||
            strlen($data['KNOrgDetails']['PhysicalReceiver']) > 30
        ) {
            log_error('OpenBox Order Create :  Invalid/Exceeded length of PhysicalReceiver');
            return false;
        }

        if (!isset($data['KNOrgDetails']['LogicalReceiver']) ||
            is_array($data['KNOrgDetails']['LogicalReceiver']) ||
            strlen($data['KNOrgDetails']['LogicalReceiver']) > 30
        ) {
            log_error('OpenBox Order Create :  Invalid/Exceeded length of LogicalReceiver');
            return false;
        }

        if (!isset($data['KNOrgDetails']['PhysicalSender']) ||
            is_array($data['KNOrgDetails']['PhysicalSender']) ||
            strlen($data['KNOrgDetails']['PhysicalSender']) > 30
        ) {
            log_error('OpenBox Order Create :  Invalid/Exceeded length of PhysicalSender');
            return false;
        }

        if (!isset($data['KNOrgDetails']['LogicalSender']) ||
            is_array($data['KNOrgDetails']['LogicalSender']) ||
            strlen($data['KNOrgDetails']['LogicalSender']) > 30
        ) {
            log_error('OpenBox Order Create :  Invalid/Exceeded length of LogicalSender');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderParties']['PartyType']) ||
            empty($data['OrderDetails']['OrderParties']['PartyType'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty OrderDetails OrderParties PartyType');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderType']) ||
            empty($data['OrderDetails']['OrderType']) ||
            strlen($data['OrderDetails']['OrderType']) > 60
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Order Type');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderProductType']) ||
            is_array($data['OrderDetails']['OrderProductType']) ||
            empty($data['OrderDetails']['OrderProductType'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty OrderProduct Type');
            return false;
        }

        if (!isset($data['OrderDetails']['ModeOfTransport']) ||
            is_array($data['OrderDetails']['ModeOfTransport']) ||
            empty($data['OrderDetails']['ModeOfTransport'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty ModeOf Transport');
            return false;
        }

        if (!isset($data['OrderDetails']['ServiceType']) ||
            is_array($data['OrderDetails']['ServiceType']) ||
            empty($data['OrderDetails']['ServiceType'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Service Type');
            return false;
        }

        if (!isset($data['OrderDetails']['DeliveryTerms']) || is_array($data['OrderDetails']['DeliveryTerms'])) {
            log_error('OpenBox Order Create :  Invalid/Empty Delivery Terms');
            return false;
        }

        if (!isset($data['OrderDetails']['GoodsValue']) || !count($data['OrderDetails']['GoodsValue'])) {
            log_error('OpenBox Order Create :  Invalid/Empty Goods values');
            return false;
        }

        if (!isset($data['OrderDetails']['GoodsValue']['Value']) || is_array($data['OrderDetails']['GoodsValue']['Value'])) {
            log_error('OpenBox Order Create :  Invalid/Empty Goods value');
            return false;
        }

        if (!isset($data['OrderDetails']['TermsOfTrade']) || !count($data['OrderDetails']['TermsOfTrade'])) {
            log_error('OpenBox Order Create :  Invalid/Empty Terms Of Trade');
            return false;
        }

        if (!isset($data['OrderDetails']['TermsOfTrade']['Incoterm']) ||
            empty($data['OrderDetails']['TermsOfTrade']['Incoterm']) ||
            strlen($data['OrderDetails']['TermsOfTrade']['Incoterm']) > 11
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Incoterm');
            return false;
        }

        if (!$data['OrderDetails']['OrderStartLocation']['PlannedDateTime']['From'] ||
            empty($data['OrderDetails']['OrderStartLocation']['PlannedDateTime']['To'])
        ) {
            log_error('OpenBox Order Create : Order start location date and time empty.');
            return false;
        }

        if (!$data['OrderDetails']['OrderEndLocation']['PlannedDateTime']['From'] ||
            empty($data['OrderDetails']['OrderEndLocation']['PlannedDateTime']['To'])
        ) {
            log_error('OpenBox Order Create : Order end location date  and time empty.');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Timezone']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Timezone'])
        ) {
            log_error('OpenBox Order Create : Invalid/Empty  Order start location Timezone');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['Timezone']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['Timezone'])
        ) {
            log_error('OpenBox Order Create : Invalid/Empty  Order end location Timezone');
            return false;
        }

        if (!is_array($data['OrderDetails']['OrderStartLocation']['DriverInstructions']) &&
            strlen($data['OrderDetails']['OrderStartLocation']['DriverInstructions']) > 1000
        ) {
            log_error('OpenBox Order Create :  Invalid Pickup Driver instructions');
            return false;
        }

        if (!is_array($data['OrderDetails']['OrderEndLocation']['DriverInstructions']) &&
            strlen($data['OrderDetails']['OrderEndLocation']['DriverInstructions']) > 1000
        ) {
            log_error('OpenBox Order Create :  Invalid Driver Delivery instructions');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['City']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['City'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Pickup City');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Postal']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Postal'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Pickup City Postal code');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Name']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Name']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['Name']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Pickup name');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Street']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Street']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['Street']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Pickup Street');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['State']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['State']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['State']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Pickup State');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['Country']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['Country'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Delivery Country');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['City']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['City'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Delivery Country');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['Postal']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['Postal'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Delivery Postal code');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['Name']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['Name'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Delivery name');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderCargoDetails']['Items']) ||
            empty($data['OrderDetails']['OrderCargoDetails']['Items'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty OrderCargoDetails Items');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderCargoSummary']['TotalQuantity']['Value']) ||
            empty($data['OrderDetails']['OrderCargoSummary']['TotalQuantity']['Value'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Total Quantity');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderCargoSummary']['TotalVolume']['Value']) ||
            empty($data['OrderDetails']['OrderCargoSummary']['TotalVolume']['Value'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Total volume');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderCargoSummary']['TotalWeight']['Value']) ||
            empty($data['OrderDetails']['OrderCargoSummary']['TotalWeight']['Value'])
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty Total weight');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['City']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['City']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['City']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty pickup city');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Street']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Street']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['Street']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty pickup street');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Postal']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Postal']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['Postal']) > 15
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty pickup postal code');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderStartLocation']['Address']['Country']) ||
            empty($data['OrderDetails']['OrderStartLocation']['Address']['Country']) ||
            strlen($data['OrderDetails']['OrderStartLocation']['Address']['Country']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty pickup country');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['City']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['City']) ||
            strlen($data['OrderDetails']['OrderEndLocation']['Address']['City']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty delivery city');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['State']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['State']) ||
            strlen($data['OrderDetails']['OrderEndLocation']['Address']['State']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty delivery State');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['Postal']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['Postal']) ||
            strlen($data['OrderDetails']['OrderEndLocation']['Address']['Postal']) > 15
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty delivery postal code');
            return false;
        }

        if (!isset($data['OrderDetails']['OrderEndLocation']['Address']['Country']) ||
            empty($data['OrderDetails']['OrderEndLocation']['Address']['Country']) ||
            strlen($data['OrderDetails']['OrderEndLocation']['Address']['Country']) > 100
        ) {
            log_error('OpenBox Order Create :  Invalid/Empty delivery country');
            return false;
        }

        if (!isset($data['OrderDetails']['DepartmentCode']) || strlen($data['OrderDetails']['DepartmentCode']) > 30) {
            log_error('OpenBox Order Create :  Invalid/Exceeded length of DepartmentCode');
            return false;
        }

        if (!isset($data['OrderDetails']['GoodsValue']['Currency']) ||
            strlen($data['OrderDetails']['GoodsValue']['Currency']) > 3
        ) {
            log_error('OpenBox Order Create :  Invalid/Exceeded length of Currency');
            return false;
        }

        $departmentCode = $data['OrderDetails']['DepartmentCode'];
        $companyCode = $data['KNOrgDetails']['CompanyCode'];
        $branchCode = $data['KNOrgDetails']['BranchCode'];
        $orderProductType = $data['OrderDetails']['OrderProductType'];
        $serviceType = $data['OrderDetails']['ServiceType'];
        $deliveryTerms = $data['OrderDetails']['DeliveryTerms'];
        $incoterm = $data['OrderDetails']['TermsOfTrade']['Incoterm'];
        $partyType = $data['OrderDetails']['OrderParties']['PartyType'];
        $modeOfTransport = $data['OrderDetails']['ModeOfTransport'];

        //* Validate Branch code, Company Code, Department code *//
        $checkCodes = $this->ci->OpenboxModel->getCount(
            'tb_department_master',
            ['department_code' => $departmentCode, 'company_code' => $companyCode, 'branch_code' => $branchCode]
        );

        if (!$checkCodes) {
            log_error('OpenBox Order Create :  Invalid department/company/branch code');
            return false;
        }
        //* Validate Product type *//
        $checkOrderType = $this->ci->OpenboxModel->getCount(
            'tb_products_master',
            ['name' => $orderProductType, 'status' => 1]
        );

        if (!$checkOrderType) {
            log_error('OpenBox Order Create :  Invalid Product type - ' . $orderProductType);
            return false;
        }

        $checkServiceType = $this->ci->OpenboxModel->getCount(
            'tb_service_master',
            ['name' => $serviceType, 'status' => 1]
        );

        if ($checkServiceType <= 0) {
            log_error('OpenBox Order Create :  Invalid Service type - ' . $serviceType);
            return false;
        }

        $checkServiceType = $this->ci->OpenboxModel->getCount(
            'tb_delivery_terms',
            ['name' => $deliveryTerms, 'status' => 1]
        );

        if ($checkServiceType <= 0) {
            log_error('OpenBox Order Create :  Invalid Delivery terms - ' . $deliveryTerms);
            return false;
        }

        $checkIncoTerms = $this->ci->OpenboxModel->getCount(
            'tb_incoterms_master',
            ['name' => $incoterm, 'status' => 1]
        );

        if (0 === $checkIncoTerms) {
            log_error('OpenBox Order Create :  Invalid Inco terms - ' . $incoterm);
            return false;
        }

        if (empty($partyType)) {
            log_error('OpenBox Order Create : Party types not found.');
            return false;
        }
        foreach ($partyType as $values) {
            $customerCode = $values['PartyID']['ID'];

            $conditions = ['code' => $customerCode, 'status' => 1];
            $checkPartyMaster = $this->ci->OpenboxModel->getCount(
                'tbl_party_master',
                $conditions
            );

            if (empty($checkPartyMaster)) {
                $saveParty = $this->ci->OpenboxModel->createParty(
                    $values,
                    $companyCode,
                    $branchCode,
                    $departmentCode,
                    $data['OrderDetails']
                );
                if (!$saveParty) {
                    log_error('OpenBox Order Create :  Unable to save new Party - ' . $customerCode);
                    return false;
                }
            }
            if (strtoupper($values['Type']) == 'CUSTOMER') {
                $customer = $this->ci->common->gettblrowdata([
                    'code' => $customerCode,
                    'name' => $values['PartyID']['Name'],
                    'status' => 1
                ], 'id', 'tb_customers', 0, 0);
                if (empty($customer)) {
                    $partyMaster = $this->ci->common->gettblrowdata($conditions, 'id', 'tbl_party_master', 0, 0);
                    $saveCustomer = $this->ci->OpenboxModel->createCustomer(
                        $values,
                        $companyCode,
                        $branchCode,
                        $partyMaster['id'],
                        $data['OrderDetails']
                    );
                    if (!$saveCustomer) {
                        log_error('OpenBox Order Create :  Unable to save new Customer - ' . $customerCode);
                        return false;
                    }
                }
            }
        }

        if (!in_array(
            $modeOfTransport,
            ['FTL', 'LTL', 'COURIER', 'AIR-LCL', 'AIR-FCL', 'SEA-LCL', 'SEA-FCL', 'GROUPAGE']
        )) {
            log_error('Edi-info: Invalid Mode Of Transport- ' . $modeOfTransport);
            return false;
        }

        return true;
    }

    public function orderData(array $data): array
    {
        $checkOrderExist = $this->ci->OpenboxModel->getCount(
            'tb_orders',
            ['external_order_id' => $data['OrderDetails']['EXTOrderID'], 'status <>' => 0]
        );

        if (!$checkOrderExist == 0 && $data['Action'] == self::ACTION_CREATE) {
            log_error('OpenBox Order Create :  Order already exist - ' . $data['OrderDetails']['EXTOrderID']);
            return [];
        }

        $createdDate = date('Y-m-d H:i:s');
        $orderTypeRowId = 0;

        $dgGoods = $data['OrderDetails']['OrderCargoSummary']['DangerousGoodsFlag'] ? 1 : 0;

        $getOrderType = $this->ci->common->gettblrowdata(
            [
                'type_name' => $data['OrderDetails']['OrderType'],
                'customer_id' => $this->ci->OpenboxModel->openBoxuserId(),
                'status' => 1,
            ],
            'id',
            'tb_order_types',
            0,
            0
        );
        if (count($getOrderType)) {
            $orderTypeRowId = $getOrderType['id'];
        }

        $transportMode = $this->ci->common->gettblrowdata(
            ['code' => $data['OrderDetails']['ModeOfTransport'], 'status' => 1],
            'id,name',
            'tb_transportmode',
            0,
            0
        );

        if (empty($transportMode)) {
            return [];
        }
        $modeofTransportId = $transportMode['id'];
        $modeOfTrasportName = $transportMode['name'];

        $orderConsineeId = $orderShipperId = $orderCustomerId = 0;
        $orderConsineeName = $orderShipperName = $orderCustomerName = "";

        foreach ($data['OrderDetails']['OrderParties']['PartyType'] as $values) {
            $customerCode = $values['PartyID']['ID'];
            $customerType = $values['Type'] ?? '0';

            switch ($customerType) {
                case 'CONSIGNEE':
                    $orderConsineeId = $values['PartyID']['ID'];
                    $orderConsineeName = $values['PartyID']['Name'];
                    break;
                case 'SHIPPER':
                    $orderShipperId = $values['PartyID']['ID'];
                    $orderShipperName = $values['PartyID']['Name'];
                    break;
                case 'CUSTOMER':
                    $orderCustomerId = $values['PartyID']['ID'];
                    $orderCustomerName = $values['PartyID']['Name'];
                    break;
                default:
                    $orderCustomerId = "";
                    $orderCustomerName = "";
            }
        }

        $checkFromTimeZone = $this->ci->common->gettblrowdata(
            ['cntry_timezone' => $data['OrderDetails']['OrderStartLocation']['Address']['Timezone'], 'status' => 1],
            'id',
            'tbl_country_master',
            0,
            0
        );
        if (count($checkFromTimeZone)) {
            $timeZoneStart = $data['OrderDetails']['OrderEndLocation']['Address']['Timezone'];
        } else {
            log_error('OpenBox Order Create : Timezone not match with eTN.');
            return [];
        }

        $checkToTimeZone = $this->ci->common->gettblrowdata(
            ['cntry_timezone' => $data['OrderDetails']['OrderEndLocation']['Address']['Timezone'], 'status' => 1],
            'id',
            'tbl_country_master',
            0,
            0
        );
        if (count($checkToTimeZone)) {
            $timeZoneEnd = $data['OrderDetails']['OrderEndLocation']['Address']['Timezone'];
        } else {
            log_error('OpenBox Order Create : Timezone not match with eTN.');
            return [];
        }

        $timeZonePickup = new DateTime($data['OrderDetails']['OrderStartLocation']['PlannedDateTime']['From']);
        $timeZonePickup->setTimezone(new DateTimeZone($timeZoneStart));
        $pickupDate = $timeZonePickup->format('Y-m-d H:i:s');

        $timeZoneDelivery = new DateTime($data['OrderDetails']['OrderEndLocation']['PlannedDateTime']['From']);
        $timeZoneDelivery->setTimezone(new DateTimeZone($timeZoneStart));
        $deliveryDate = $timeZoneDelivery->format('Y-m-d H:i:s');

        $timeZonePickupEnd = new DateTime($data['OrderDetails']['OrderStartLocation']['PlannedDateTime']['To']);
        $timeZonePickupEnd->setTimezone(new DateTimeZone($timeZoneEnd));
        $pickupDateEnd = $timeZonePickupEnd->format('Y-m-d H:i:s');

        $timeZoneDeliveryEnd = new DateTime($data['OrderDetails']['OrderEndLocation']['PlannedDateTime']['To']);
        $timeZoneDeliveryEnd->setTimezone(new DateTimeZone($timeZoneEnd));
        $deliveryDateEnd = $timeZoneDeliveryEnd->format('Y-m-d H:i:s');

        $customerRowId = 0;
        $customer = $this->ci->common->gettblrowdata(
            ['code' => $orderCustomerId],
            'id',
            'tb_customers',
            0,
            0
        );
        if (!empty($customer)) {
            $customerRowId = $customer['id'];
        }

        return [
            'external_order_id' => $data['OrderDetails']['EXTOrderID'],
            'goods_value' => $data['OrderDetails']['GoodsValue']['Value'],
            'currency' => $data['OrderDetails']['GoodsValue']['Currency'],
            'serviceType' => $data['OrderDetails']['ServiceType'],
            'departmentCode' => $data['OrderDetails']['DepartmentCode'],
            'incoterm' => $data['OrderDetails']['TermsOfTrade']['Incoterm'],
            'orderTypeRowId' => $orderTypeRowId,
            'shipper_id' => $orderShipperId,
            'pickup_custid' => $orderShipperId,
            'pickup_partyid' => $orderShipperId,
            'drop_custid' => $orderConsineeId,
            'drop_partyid' => $orderConsineeId,
            'customer_id' => $customerRowId,
            'orderCustomerId' => $orderCustomerId,
            'orderCustomerName' => $orderCustomerName,
            'pickup_country' => $data['OrderDetails']['OrderStartLocation']['Address']['Country'],
            'pickup_city' => $data['OrderDetails']['OrderStartLocation']['Address']['City'],
            'pickup_pincode' => $data['OrderDetails']['OrderStartLocation']['Address']['Postal'],
            'pickup_company' => $orderShipperName,
            'pickup_address1' => $data['OrderDetails']['OrderStartLocation']['Address']['Name'],
            'pickup_address2' => $data['OrderDetails']['OrderStartLocation']['Address']['Street'],
            'delivery_country' => $data['OrderDetails']['OrderEndLocation']['Address']['Country'],
            'delivery_city' => $data['OrderDetails']['OrderEndLocation']['Address']['City'],
            'delivery_pincode' => $data['OrderDetails']['OrderEndLocation']['Address']['Postal'],
            'delivery_company' => $orderConsineeName,
            'delivery_address1' => $data['OrderDetails']['OrderEndLocation']['Address']['Name'],
            'quantity' => $data['OrderDetails']['OrderCargoSummary']['TotalQuantity']['Value'],
            'volume' => $data['OrderDetails']['OrderCargoSummary']['TotalVolume']['Value'],
            'weight' => $data['OrderDetails']['OrderCargoSummary']['TotalWeight']['Value'],
            'pickup_datetime' => $pickupDate,
            'pickup_endtime' => $pickupDateEnd,
            'delivery_datetime' => $deliveryDate,
            'drop_endtime' => $deliveryDateEnd,
            'modeofTransportId' => $modeofTransportId,
            'dgGoods' => $dgGoods,
            'modeOfTrasportName' => $modeOfTrasportName,
            'user_id' => $this->ci->OpenboxModel->openBoxAdminUseId(),
            'company_code' => $data['KNOrgDetails']['CompanyCode'],
            'branch_code' => $data['KNOrgDetails']['BranchCode'],
            'product' => $data['OrderDetails']['OrderProductType'],
            'KNOrgDetails' => $data['KNOrgDetails'],
            'OrderDetails' => $data['OrderDetails'],
            'status' => 1,
            'is_created' => 1,
            'createdon' => $createdDate,
            'dropState' => $data['OrderDetails']['OrderEndLocation']['Address']['State'],
            'PhysicalReceiver' => $data['KNOrgDetails']['PhysicalReceiver'] ?? '',
            'LogicalReceiver' => $data['KNOrgDetails']['LogicalReceiver'] ?? '',
            'PhysicalSender' => $data['KNOrgDetails']['PhysicalSender'] ?? '',
            'LogicalSender' => $data['KNOrgDetails']['LogicalSender'] ?? '',
            'SequenceNumber' => $data['Header']['SequenceNumber'],
            'Action' => $data['Action'],
            'created_source' => (new ordercreationsourcehelper())->getOrderSourceId(OrderCreationSourceHelper::OPENBOXES)
        ];
    }
}
