<?php

class RussianIntegrationService
{
    protected $ci;

    protected const PICKUP_GATE_IN = 'pickup_gate_in';
    protected const PICKUP_GATE_OUT = 'pickup_gate_out';
    protected const PICKUP = 'pickup';
    protected const IN_TRANSIT = 'in_transit';
    protected const DELIVERY_GATE_IN = 'delivery_gate_in';
    protected const DELIVERY_GATE_OUT = 'delivery_gate_out';
    protected const DELIVERY = 'delivery';

    protected const STOP_STATUS_MAPPING = [];

    protected $stopStatuses = [];
    protected $logPrefix;

    public function __construct()
    {
        $this->ci =& get_instance();
        $this->ci->load->helper('log');

        $this->addStopStatus(self::PICKUP_GATE_IN, 'P', 2, '0420', 'Pickup Gate In')
            ->addStopStatus(self::PICKUP_GATE_OUT, 'P', 2, '0191', 'Pickup Gate Out')
            ->addStopStatus(self::PICKUP, 'P', 1, '0500', 'Pickup')
            ->addStopStatus(self::IN_TRANSIT, 'P', 1, '1550', 'In Transit')
            ->addStopStatus(self::DELIVERY_GATE_IN, 'D', 2, '0192', 'Delivery Gate In')
            ->addStopStatus(self::DELIVERY_GATE_OUT, 'D', 2, '3000', 'Delivery Gate Out')
            ->addStopStatus(self::DELIVERY, 'D', 1, '2300', 'Delivery');
    }

    protected function addStopStatus(string $key, string $stopType, int $statusId, string $statusCode, string $statusStage): self
    {
        $this->stopStatuses = array_merge($this->stopStatuses, [
                $key => [
                    'stop_type' => $stopType,
                    'status_id' => $statusId,
                    'status' => 1,
                    'status_code' => $statusCode,
                    'status_stage' => $statusStage
                ]
            ]
        );

        return $this;
    }

    protected function getStopStatuses(): array
    {
        return $this->stopStatuses;
    }

    protected function findEmployeeByBookingId(int $bookingId): ?object
    {
        $employees = $this->ci->db
            ->select('drop_stopid, id')
            ->from('tb_employee')
            ->where([
                'order_id' => $bookingId
            ])->get()->result();

        return count($employees) === 1 ? $employees[0] : null;
    }

    protected function findVehicleByShiftId(int $shiftId): ?object
    {
        $vehicles = $this->ci->db
            ->select('vehicle_id')
            ->from('tb_shft_veh')
            ->where(['shft_id' => $shiftId, 'status' => 1])
            ->get()->result();

        return count($vehicles) === 1 ? $vehicles[0] : null;
    }

    protected function findTruckDriverByVehicleId(int $vehicleId): ?object
    {
        $drivers = $this->ci->db
            ->select('td.id')
            ->from('tb_truck_drivers td')
            ->join('tbl_assigned_drivers ad', 'td.id = ad.driver_id', 'left')
            ->where(['ad.vehicle_id' => $vehicleId, 'ad.status' => 1, 'td.status' => 1])
            ->get()->result();

        return count($drivers) === 1 ? $drivers[0] : null;
    }

    protected function validateStatus($status): bool
    {
        return array_key_exists($status, static::STOP_STATUS_MAPPING);
    }

    protected function validateXmlFields(array $xml): array
    {
        $errors = [];
        foreach ($xml as $field => $value) {
            if (empty($value)) {
                $errors[] = $this->logPrefix . ' - Invalid value for ' . $field;
            }
        }

        return $errors;
    }

    /**
     * Retrieve the XML and convert it to an associative array;
     * If invalid it returns false
     *
     * @param string $xml
     * @return false|array
     */
    protected function retrieveXml(string $xml)
    {
        if (!$result = simplexml_load_string($xml)) {
            log_error($this->logPrefix . ' - Incorrect XML returned: ' . $xml);
            return false;
        }

        $result = json_decode(json_encode($result), true);
        if (isset($result['error'])) {
            log_error($this->logPrefix . ' - Error processing retrieved XML: ' . $result['error']);
            return false;
        }

        return $result;
    }
}
