<?php

namespace App\Libraries;

use App\Helpers\ParsedOrderIdHelperInterface;
use App\Models\OrderIdModelInterface;

class UniqueOrderIdGeneratorService
{
    private $orderModel;
    private $parseOrderIdHelper;

    public function __construct(OrderIdModelInterface $orderModel, ParsedOrderIdHelperInterface $parseOrderIdHelper)
    {
        $this->orderModel = $orderModel;
        $this->parseOrderIdHelper = $parseOrderIdHelper;
    }


    /**
     * @param $orderId
     * @param int $tries
     * @return int|null
     * @throws \Exception
     */

    public function getUniqueOrderId($orderId, $tries = 0): ?int
    {
        //log_message("error","22:".$orderId);
        $parsedOrderId = $this->parseOrderIdHelper->parseOrderId($orderId);
        
        if (!$this->isUnique($parsedOrderId)) {
            $orderId = $this->nextUniqueOrderId($parsedOrderId);
        }
        //log_message("error","33:".$orderId);
        try {
            $this->orderModel->saveOrderId($orderId);
        } catch (\Exception $e) {
            if ($tries === 2) {
                throw new \Exception('Maximum retries for generating a unique orderId reached');
            }
            $tries++;
            $orderId = $this->getUniqueOrderId(($orderId), $tries);
            //log_message("error","44:".$orderId);
        }
        return (int) $orderId;
    }

    private function nextUniqueOrderId(ParsedOrderIdHelperInterface $orderId): int
    {
        $id = $orderId->getFullId();
        $highestOrderId = $this->parseOrderIdHelper->parseOrderId($this->orderModel->getMaxOrderId($orderId));
        if($highestOrderId->getFullId() < $id) {
            $highestOrderId = $this->parseOrderIdHelper->parseOrderId($id);
        }

        return $highestOrderId->upIdCode();
    }

    private function isUnique(ParsedOrderIdHelperInterface $orderId): bool
    {
        return $this->orderModel->getNumberOfOrderIDs($orderId) < 1;
    }
}

