<?php

namespace SVKAPI\v1;

defined('BASEPATH') OR exit('No direct script access allowed');

use SVKAPI\v1\Exceptions\ValidationException;

class DocumentUploadHandler extends BaseInboundHandler
{
    protected $modelClassName = 'SVKAPI\v1\Models\DocumentUpload';
    private $documentData = [];

    public function __construct()
    {
        parent::__construct();
        $this->CI->load->model([
            'common',
            'document_type',
            'document_uploads',
            'order',
            'stop_detail',
            'trips_model',
        ]);
        $this->CI->load->helper('file');
        $this->CI->load->library(['uuid','Edi_logger']);
    }

    public function processMessage(): void
    {
        $this->deserializeData();
        $this->validateData();
        $this->writeToDisk();
        $this->populateDatabase();
    }

    protected function validateData(): void
    {
        parent::validateData();

        $this->log('Veryfing data integrity');

        $documentType = $this->model->getType();
        $tripId = $this->model->getTripId();
        $orderId = $this->model->getOrderId();
        $documentTypeResults = $this->CI->document_type->select([
            'id',
        ], [
            'type_name' => $documentType,
        ]);
        if (!$documentTypeResults->num_rows()) {
            throw new ValidationException("Document type {$documentType} is not supported");
        }
        // save necessary data for later
        $this->documentData['doc_type'] = $documentTypeResults->result_array()[0]['id'];

        try {
            $shiftRowId = $this->CI->trips_model->get_id_from_shipmentid($tripId); // table: tb_shifts
            $orderRowId = $this->CI->order->get_id_from_order_number($orderId); // table: tb_orders
        } catch (\Exception $e) {
            throw new ValidationException($e->getMessage());
        }
        if (!$this->CI->trips_model->is_order_on_shift($shiftRowId, $orderId, $orderRowId)) {
            throw new ValidationException("Order {$orderId} is not on Trip {$tripId}");
        }
        // save necessary data for later
        $this->documentData['trip_id'] = $shiftRowId;
        $this->documentData['shipment_id'] = $shiftRowId;
        $this->documentData['order_id'] = $orderRowId;
        $this->CI->edi_logger->setTxn_obj_id($orderRowId);
        $orderInfo = $this->CI->common->gettblrowdata(['id' => $orderRowId], "user_id,company_code,branch_code", "tb_orders", 0, 0);
        $this->CI->edi_logger->setUser_id($orderInfo['user_id']);
        $this->CI->edi_logger->setCompany_code($orderInfo['company_code']);
        $this->CI->edi_logger->setBranch_code($orderInfo['branch_code']);

        $stopDetailsResults = $this->CI->stop_detail->select([
            'id',
            'drop_stopid',
        ], [
            'order_id' => $orderId,
            'shift_id' => $shiftRowId,
        ], 1);
        if (!$stopDetailsResults->num_rows()) {
            throw new ValidationException('Missing Stop Details');
        }
        // save necessary data for later
        $this->documentData['stop_id'] = $stopDetailsResults->result_array()[0]['drop_stopid'];
        $this->documentData['stop_detail_id'] = $stopDetailsResults->result_array()[0]['id'];
        $this->documentData['stop_type'] = 'D';
    }

    protected function writeToDisk(): void
    {
        $filename = $this->model->getFilename();
        $pathinfo = pathinfo($filename);
        $filenameWithDate = sprintf('%s_%s.%s', $pathinfo['filename'], date('Y-m-d_H-i-s'), $pathinfo['extension']);
        $path = NEW_POD_PATH . $filenameWithDate;
        $content = base64_decode($this->model->getContent());
        if (!write_file($path, $content)) {
            throw new \RuntimeException("Failed to write Document {$filename} to disk");
        }
        // save necessary data for later
        $this->documentData['imgpath'] = $filenameWithDate;
        if ($pathinfo['extension'] === 'pdf') {
            $this->documentData['pdfpath'] = $filenameWithDate;
        }
        $this->documentData['filesize'] = filesize($path);
    }

    protected function populateDatabase(): void
    {
        $filename = $this->model->getFilename();

        $this->log("Saving Document {$filename}...");

        $this->documentData['user_id'] = $this->userId;
        $this->documentData['status'] = 1;
        $this->documentData['createdon'] = date('Y-m-d H:i:s');
        $this->documentData['hash'] = $this->CI->uuid->v4();
        if (!$this->CI->document_uploads->insert($this->documentData)) {
            throw new \RuntimeException("Failed to write Document {$filename} to database");
        }

        $this->log("Saving Document {$filename} DONE!");
    }
}
