<?php

namespace SVKAPI\v1;

defined('BASEPATH') OR exit('No direct script access allowed');

use SVKAPI\v1\Exceptions\ValidationException;
use SVKAPI\v1\Models\Status as Status;

class StatusExecutionHandler extends BaseInboundHandler
{
    protected $modelClassName = 'SVKAPI\v1\Models\StatusExecution';

    protected $statusMap = [
        '0500' => ['KN002', 'KN007', 'KN001'],
        '1550' => ['KN002', 'KN007', 'KN001', 'KN005'],
        '3000' => ['KN002', 'KN007', 'KN001']
    ];

    protected $statusCodeMap = [
        'P' => [
            'KN002' => '0420',
            'KN007' => '0500',
            'KN001' => '0191',
            'KN005' => '1550'
        ],
        'D' => [
            'KN002' => '0192',
            'KN007' => '2300',
            'KN001' => '3000'
        ]
    ];

    public function __construct()
    {
        parent::__construct();
        $this->CI->load->model([
            'order',
            'order_references',
            'order_status',
            'ship_order_stops',
            'status_master_model',
            'stop_detail',
            'stop_status',
            'trip_employee',
            'trips_carrier_details',
            'trips_model',
            'common'
        ]);
        $this->CI->load->library('Edi_logger');
    }

    public function processMessage(): void
    {

        $this->deserializeData();
        $this->validateData();
        $this->populateDatabase();

    }

    protected function validateData(): void
    {
        parent::validateData();

        $this->log('Veryfing data integrity');

        $tripId = $this->model->getTripId();
        $statuses = $this->model->getStatuses();

        if (!$this->CI->trips_model->exists(['shipmentid' => $tripId])) {
            throw new ValidationException("Trip {$tripId} does not exist");
        }


        foreach ($statuses as $status) {
            $orderId = $status->getOrderId();

            if (!$this->CI->order->exists(['order_id' => $orderId])) {
                throw new ValidationException("Order {$orderId} does not exist");
            }

            try {
                $shiftRowId = $this->CI->trips_model->get_id_from_shipmentid($tripId); // table: tb_shifts
                $orderRowId = $this->CI->order->get_id_from_order_number($orderId); // table: tb_orders
            } catch (\Exception $e) {
                throw new ValidationException($e->getMessage());
            }

            if (!$this->CI->trips_model->is_order_on_shift($shiftRowId, $orderId, $orderRowId)) {
                throw new ValidationException("Order {$orderId} is not on Trip {$tripId}");
            }
        }
    }

    protected function processStatus(Status $status, string $stopType): void
    {
        $tripId = $this->model->getTripId();
        $shiftRowId = $this->CI->trips_model->get_id_from_shipmentid($tripId); // table: tb_shifts
        $tripRowId = $this->CI->trips_carrier_details->get_id_from_shift_id($shiftRowId); // table: tb_trips

        $orderId = $status->getOrderId();
        $this->CI->edi_logger->setTxn_obj_id($orderId);
        $orderInfo = $this->CI->common->gettblrowdata(['order_id' => $orderId], "user_id,company_code,branch_code", "tb_orders", 0, 0);
        $this->CI->edi_logger->setUser_id($orderInfo['user_id'] ?? 2);
        $this->CI->edi_logger->setCompany_code($orderInfo['company_code']);
        $this->CI->edi_logger->setBranch_code($orderInfo['branch_code']);
        $orderRowId = $this->CI->order->get_id_from_order_number($orderId); // table: tb_orders
        $executionDateTime = $status->getExecutionDateTime();
        $statusCode = $status->getStatusCode();

        $coordinates = $status->getCoordinates();

        $this->log("Saving Status for Trip {$tripId} Order {$orderId}...");

        $this->CI->db->trans_begin();

        // table: tb_employee
        $stopDetailsConditions = [
            'order_id' => $orderId,
            'shift_id' => $shiftRowId,
        ];
        $stopDetails = $this->CI->stop_detail->find($stopDetailsConditions);
        $stopDetailsRowId = $stopDetails[0]->id;
        $pickupStopRowId = $stopDetails[0]->stop_id;
        $dropStopRowId = $stopDetails[0]->drop_stopid;

        $statusMasterRowId = $this->CI->status_master_model->get_model_id_from_kn_status_code($statusCode);
        if (is_null($statusMasterRowId)) {
            throw new ValidationException("Invalid Status Code {$statusCode}");
        }
        if (is_null($stopType)) {
            throw new ValidationException("Unknown stop type for Status Code {$statusCode}");
        }
        $stopRowId = $stopType === 'P' ? $pickupStopRowId : $dropStopRowId;

        if (isset($this->statusCodeMap[$stopType][$statusCode])) {
            $statusCode = $this->statusCodeMap[$stopType][$statusCode];
        }

        // table: tb_order_status
        $orderStatusConditions = [
            'order_id' => $orderRowId,
            'status_id' => $statusMasterRowId,
            'status_code' => $statusCode,
            //'status_date' => $executionDateTime->format('Y-m-d H:i:s'),
        ];
        $orderStatusData = [
            'status_type' => NULL,
            'shift_id' => $shiftRowId,
        ];
        if ($coordinates) {
            $orderStatusData['latitude'] = $coordinates->getLatitude();
            $orderStatusData['longitude'] = $coordinates->getLongitude();
        }
        // if ($isOrderStatus) {
        //     $this->CI->order_status->insert_or_update(
        //         array_merge($orderStatusConditions, $orderStatusData),
        //         $orderStatusConditions
        //     );
        // }

        // table: tb_stop_status
        $stopStatusConditions = [
            'order_id' => $orderRowId,
            'stop_id' => $stopRowId,
            'status_id' => $statusMasterRowId,
            'status_code' => $statusCode,
        ];
        $stopStatusData = [
            'shipment_id' => $shiftRowId,
            'stop_detail_id' => $stopDetailsRowId,
            'stop_type' => $stopType,
            'trip_id' => $tripRowId,
            'reason' => 'From StatusExecution API',
            'vehicle_id' => NULL,
            'driver_id' => NULL,
            'createdon' => $executionDateTime->format('Y-m-d H:i:s'),
        ];
        if ($coordinates) {
            $stopStatusData['latitude'] = $coordinates->getLatitude();
            $stopStatusData['longitude'] = $coordinates->getLongitude();
        }
        $this->CI->stop_status->insert_or_update(
            array_merge($stopStatusConditions, $stopStatusData),
            $stopStatusConditions
        );

        // table: tb_trip_employee
        $tripEmployeeConditions = [
            'employee_id' => $stopDetailsRowId,
            'stop_id' => $stopRowId,
            'trip_id' => $tripRowId,
            'status' => 1,
        ];
        $tripEmployeeData = [
            'stime' => $executionDateTime->format('Y-m-d H:i:s'),
            'check_in' => $executionDateTime->format('Y-m-d H:i:s'),
            'absent_reason' => 'Updated by StatusExecution API',
            'pd_status' => 1,
        ];
        $this->CI->trip_employee->insert_or_update(
            array_merge($tripEmployeeConditions, $tripEmployeeData),
            $tripEmployeeConditions
        );

        $references = [
            'AWB' => $status->getCarrierAWB(),
            'SIGNATURE' => $status->getSignatureName(),
        ];
        foreach ($references as $referenceName => $referenceValue) {
            // table: tb_order_references
            $this->CI->order_references->insert_or_update([
                'order_id' => $orderRowId,
                'reference_id' => $referenceName,
                'ref_value' => $referenceValue,
            ], [
                'order_id' => $orderRowId,
                'reference_id' => $referenceName,
            ]);
        }

        if ($this->CI->db->trans_status() === false) {
            $this->CI->db->trans_rollback();
            throw new \RuntimeException('Database transaction rollback');
        }
        $this->CI->db->trans_commit();

        $this->log("Saving Status for Trip {$tripId} Order {$orderId} DONE!");
    }


    protected function populateDatabase(): void
    {
        $statuses = $this->model->getStatuses();
        foreach ($statuses as $status) {
            $statusCode = $status->getStatusCode();
            $stopType = $this->getStopTypeFromStatusCode($statusCode);

            if (in_array($statusCode, array_keys($this->statusMap))) {
                foreach ($this->statusMap[$statusCode] as $mapCode) {
                    $status->setStatusCode($mapCode, $stopType);
                    $this->processStatus($status, $stopType);
                }
            } else {
                $this->processStatus($status, $stopType);
            }
        }
    }

    protected function getStopTypeFromStatusCode($statusCode): ?string
    {
        $statuses = [
            '0022' => 'P',
            '0062' => 'D',
            '0100' => 'D',
            '0120' => 'D',
            '0142' => 'P',
            '0300' => 'P',
            '0400' => 'P',
            '0490' => 'P',
            '0500' => 'P',
            '1000' => 'P',
            '1010' => 'D',
            '1052' => 'D',
            '1075' => 'D',
            '1130' => 'P',
            '1200' => 'D',
            '1270' => 'P',
            '1290' => 'P',
            '1300' => 'P',
            '1327' => 'D',
            '1375' => 'D',
            '1380' => 'P',
            '1385' => 'D',
            '1400' => 'P',
            '1410' => 'P',
            '1450' => 'D',
            '1520' => 'P',
            '1550' => 'P',
            '1564' => 'D',
            '1577' => 'D',
            '1579' => 'D',
            '1640' => 'D',
            '2210' => 'D',
            '2333' => 'D',
            '2340' => 'D',
            '2400' => 'D',
            '2500' => 'D',
            '2510' => 'D',
            '2525' => 'P',
            '2533' => 'D',
            '2600' => 'D',
            '2990' => 'D',
            '3000' => 'D',
            '3001' => 'D',
            '3021' => 'D',
            '3033' => 'D',
            '3040' => 'D',
            '3051' => 'D',
            '4210' => 'D',
            '5025' => 'D',
            '8653' => 'D',
            '8800' => 'D',
            '8850' => 'D',
            '9610' => 'P',
            '9800' => 'D',
        ];
        return $statuses[$statusCode] ?? null;
    }
}
