<?php
namespace SVKAPI\v1;

use SVKAPI\v1\Clients\ApiClient;
use SVKAPI\v1\Mappings\TripMapping;
use SVKAPI\v1\Mappings\OrdersMapping;
use SVKAPI\v1\Mappings\ReferencesMapping;
use SVKAPI\v1\Mappings\TripGoodsMapping;
use SVKAPI\v1\Mappings\XmlOutput;
use SVKAPI\v1\Models\Trip;
use SVKAPI\v1\Exceptions\MissingConfigException;
use SVKAPI\v1\Exceptions\InvalidXMLException;
use SVKAPI\v1\Exceptions\MissingDataException;

class TripNotification
{
    /**
     * @var int
     */
    private $tripId;

    /**
     * @var Trip
     */
    private $tripModel = null;

    private $edilogid = null;

    /**
     * @var TripMapping
     */
    private $tripMapping;

    /**
     * @var TripGoodsMapping
     */
    private $tripGoodsMapping;

    /**
     * @var object
     */
    private $CI;


    /**
     * @param int $tripId
     * @return string|null
     */
    public function process(int $tripId): ?string
    {
        $this->CI = &get_instance();
        $this->CI->load->library('Edi_logger');
        $this->tripId = $tripId;
        $this->tripModel = new Trip();

        $messageID = "{$tripId}-" . time();
        $this->tripModel->setMessageId($messageID);
        $statusmsg = "";
        try {

            $this->mapModels();
            $this->generateOutput();
            $this->sendRequest();
            $response = $this->CI->db->set('is_carrier_notified', 1)
                ->where('id', $tripId)
                ->update('tb_shifts');
            $this->CI->edi_logger->setEdi_request($this->xmlOutput->getOutput());
            if (!empty($response)) {
                $statusmsg = "Success";
            } else {
                $statusmsg = "Failed for unformatted request,Please try again!";
            }

        } catch (\Exception $e) {
            log_message("error", "Carrier Integration MessageID: {$messageID} - " . $e->getMessage());
            throw new \Exception($e->getMessage());
            $statusmsg = "Catch Block:" . "Carrier Integration MessageID: {$messageID} - " . $e->getMessage();

        } finally {
            //log_message("error", "EDI Status" . $statusmsg);
            $this->CI->edi_logger->setEdi_type(1);
            $this->CI->edi_logger->setTransaction_id(time());
            $this->CI->edi_logger->setEdi_id("34");
            $this->CI->edi_logger->setEdi_name('Shipsy');
            $this->CI->edi_logger->setBounded_type(1);
            $this->CI->edi_logger->setEdi_format_type('XML');
            $this->CI->edi_logger->setStatus(($statusmsg == 'Success' ? 1 : 0));
            $this->CI->edi_logger->setEdi_response($this->apiClient->getResponseBody());
            $this->CI->edi_logger->setObj_type_name('Shipsy Order');
            $this->CI->edi_logger->saveToEdiLogs();
        }
        return $this->apiClient->getResponseBody() . '<br/>' . $this->xmlOutput->getOutput();
    }

    /**
     * @throws MissingDataException
     */
    private function mapModels(): void
    {
        $this->tripMapping = new TripMapping();
        $this->tripGoodsMapping = new TripGoodsMapping();
        $this->referencesMapping = new ReferencesMapping();
        $this->ordersMapping = new OrdersMapping();

        $this->tripMapping->map($this->tripId, $this->tripModel);
        $this->tripGoodsMapping->map($this->tripId, $this->tripModel);
        $this->referencesMapping->map($this->tripId, $this->tripModel);
        $this->ordersMapping->map($this->tripId, $this->tripModel);
    }

    /**
     * @throws InvalidXMLException
     */
    private function generateOutput(): void
    {
        $this->xmlOutput = new XmlOutput();
        $this->xmlOutput->generate($this->tripModel);
    }

    /**
     * @return bool
     * @throws MissingConfigException
     * @throws InvalidResponseException
     */
    private function sendRequest(): bool
    {

        if (!defined('SHIPSY_AUTHORIZATION') or !defined('SHIPSY_TRIP_ENDPOINT')) {
            throw new MissingConfigException('Please set up SHIPSY_AUTHORIZATION and SHIPSY_TRIP_ENDPOINT for current environment.');
        }

        $this->apiClient = new ApiClient();
        return $this->apiClient->setEndpoint(SHIPSY_TRIP_ENDPOINT)
            ->setMethod('POST')
            ->setData($this->xmlOutput->getOutput())
            ->setOptions([
                'headers' => [
                    'api-key' => SHIPSY_AUTHORIZATION,
                    'Content-Type' => 'application/xml',
                ],
            ])->send();


    }

    /**
     * @return void
     * @throws Exception
     */
    private function sendDebugEmail(): void
    {
        if (!defined('SHIPSY_DEBUG_EMAILS')) {
            return;
        }

        $CI = &get_instance();
        $CI->load->library('email');
        $CI->email->to(SHIPSY_DEBUG_EMAILS); // comma separated
        $CI->email->from('svkonekt@kuehne-nagel.com', 'svkonekt');
        $CI->email->subject("Carrier Integration MessageID: {$this->tripModel->getMessageId()}");
        $CI->email->set_mailtype('text');
        $CI->email->message($this->xmlOutput->getOutput());
        if (!$CI->email->send()) {
            throw new \Exception('Unable to send debug email');
        }
        $CI->email->clear(true);
    }

    /**
     * @return void
     */
    private function saveMessage(): void
    {
        $CI = &get_instance();
        $CI->load->helper('file');
        $content = date('Y-m-d H:i:s') . "\nCarrier Integration MessageID: {$this->tripModel->getMessageId()}\n\n{$this->xmlOutput->getOutput()}\n\n";
        write_file('./assets/tmp/shipsy.log', $content, 'a');
    }


}
