<?php
defined('BASEPATH') or exit('No direct script access allowed');

class InstakartEDIServices
{
    private $CI;
    private $carrier = 'INSTAKART';
    private $appKey  = '';
    private $apiKey  = '';
    private $baseUrl = '';
    private $code    = '';

    public function __construct()
    {
        $this->CI = &get_instance();

        $this->CI->load->model(['Singlestatusmodel', 'Order']);
        $this->CI->load->library(['Edi_logger', 'uuid', 'apis/FlipkartEDIServices']);
        $this->CI->load->helper('user_helper');

        $query = $this->CI->db->get_where(
            'tb_auth_tokens',
            ['token_for' => $this->carrier]
        );

        $tokenRow = $query->row_array();

        $this->appKey  = $tokenRow['auth_token'] ?? '';
        $this->apiKey  = $tokenRow['api_key'] ?? '';
        $this->baseUrl = $tokenRow['base_url'] ?? '';
        $this->code    = $tokenRow['code'] ?? '';
    }

    public function createOrder(array $orderData, string $orderId, int $id): void
    {
        $pickupDate = $orderData['sourceInfo']['EstimatedDateTime']['From']['DateTime'] ?? '';

        if (empty($pickupDate) || $pickupDate === '0000-00-00 00:00:00') {
            $pickupDate = date('Y-m-d H:i:s');
        }

        $dt = new DateTime($pickupDate);

        $pickup = $orderData['sourceInfo'] ?? [];
        $drop   = $orderData['destinationInfo'] ?? [];

        $pickup_name    = $pickup['Company']['Name'] ?? '';
        $pickup_street  = $pickup['Address']['Address2'] ?? '';
        $pickup_city    = $pickup['Address']['City'] ?? '';
        $pickup_state   = $pickup['Address']['State'] ?? '';
        $pickup_pincode = $pickup['Address']['Postal'] ?? '';

        $drop_name    = $drop['Company']['Name'] ?? '';
        $drop_street  = $drop['Address']['Address2'] ?? '';
        $drop_city    = $drop['Address']['City'] ?? '';
        $drop_state   = $drop['Address']['State'] ?? '';
        $drop_pincode = $drop['Address']['Postal'] ?? '';

        $thuData = $orderData['cargoDetails']['CargoThuDetails'] ?? [];
        $thuData = isset($thuData[0]) ? $thuData : [$thuData];

        $lbhData     = [];
        $totalLength = $totalWidth = $totalHeight = $totalWeight = $totalQuantity = 0;

        foreach ($thuData as $i => $cargo) {

            $length   = (float) ($cargo['Length'] ?? 0);
            $width    = (float) ($cargo['Width'] ?? 0);
            $height   = (float) ($cargo['Height'] ?? 0);
            $weight   = (float) ($cargo['Weight'] ?? 0);
            $quantity = (int) ($cargo['Quantity'] ?? 0);

            $totalLength += $length;
            $totalWidth += $width;
            $totalHeight += $height;
            $totalWeight += $weight;
            $totalQuantity += $quantity;

            $lbhData[] = [
                "packetCount"         => $quantity,
                "packetLength"        => $this->formatNumber($length),
                "packetWidth"         => $this->formatNumber($width),
                "packetHeight"        => $this->formatNumber($height),
                "packetNo"            => null,
                "customerPacketRefNo" => $orderId . '-' . ($i + 1),
                "actualWeight"        => $this->formatNumber($weight),
                "invoiceNo"           => null,
            ];
        }
        $tknRef = getorderreftypeinfo('TKN', $id);

        if (empty($tknRef)) {
            $responseData    = [];
            $serverDocketURL = $this->baseUrl . 'order/create';
            try {
                $ch = curl_init($serverDocketURL);

                curl_setopt_array($ch, [
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_FOLLOWLOCATION => true,
                    CURLOPT_TIMEOUT        => 30,
                    CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
                    CURLOPT_HTTPHEADER     => [
                        'Accept: application/json',
                        'Content-Type: application/json',
                        'Authorization: Bearer ' . $this->appKey,
                    ],
                ]);

                $response = curl_exec($ch);

                if ($response === false) {
                    throw new Exception('cURL Error: ' . curl_error($ch));
                }

                $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);

                if ($httpCode !== 200) {
                    throw new Exception('Unexpected HTTP Code: ' . $httpCode);
                }

                $apiResponse = json_decode($response, true);

                if (
                    json_last_error() === JSON_ERROR_NONE &&
                    ! empty($apiResponse['data'][0])
                ) {
                    $docketNo = $apiResponse['data'][0];
                } else {
                    log_message(
                        'error',
                        'Invalid Instakart Docket response: ' . $response
                    );
                }

            } catch (Throwable $e) {
                log_message('error', 'Instakart Docket API Error: ' . $e->getMessage());
            }
        }

        if (! empty($docketNo)) {
            $this->CI->db->insert('tb_order_references', [
                'order_id'     => $id,
                'reference_id' => 'TKN',
                'ref_value'    => $docketNo,
                'createdon'    => date('Y-m-d H:i:s'),
            ]);

            $payload = [
                "poNumber"              => (string) $orderId,
                "travelMode"            => "Road",
                "grossWeight"           => $this->formatNumber($totalWeight < 1 ? 1 : $totalWeight),
                "packetCount"           => $totalQuantity,
                "lbhData"               => $lbhData,
                "invoiceDetails"        => [[
                    "invoiceNo"     => (string) $orderId,
                    "invoiceAmount" => 1,
                    "invoiceDate"   => date('d-m-Y'),
                ]],
                "consignor"             => [
                    "consignorCode"    => null,
                    "consignorPincode" => trim($pickup_pincode ?: ""),
                    "consignorName"    => trim($pickup_name ?: ""),
                    "address1"         => trim($pickup_street ?: ""),
                    "city"             => trim($pickup_city ?: ""),
                    "state"            => trim($pickup_state ?: ""),
                    "contactName"      => null,
                    "contactPhoneno"   => null,
                    "email"            => "",
                ],
                "consignee"             => [
                    "consigneeCode"    => null,
                    "consigneePincode" => trim($drop_pincode ?: ""),
                    "consigneeName"    => trim($drop_name ?: ""),
                    "address1"         => trim($drop_street ?: ""),
                    "city"             => trim($drop_city ?: ""),
                    "state"            => trim($drop_state ?: ""),
                    "contactName"      => null,
                    "contactPhoneno"   => null,
                    "email"            => null,
                ],
                "docketNo"              => $docketNo,
                "packetLbhUom"          => "IN",
                "totalConsignmentValue" => 1,
                "ftlOrPtl"              => "0",
                "openBoxPickup"         => 0,
            ];

            $vendor      = $this->CI->db->get_where('tb_vendors', ['code' => $this->code])->row_array();
            $companyCode = $vendor['company_code'] ?? '';
            $branchCode  = $vendor['branch_code'] ?? '';
            $userId      = $vendor['user_id'] ?? '';

            $responseData = [];

            try {
                $serverURL = $this->baseUrl . 'order/create';

                $ch = curl_init($serverURL);
                curl_setopt_array($ch, [
                    CURLOPT_HTTPHEADER     => [
                        'Content-Type: application/json',
                        "Authorization: Bearer $this->appKey",
                    ],
                    CURLOPT_POST           => true,
                    CURLOPT_POSTFIELDS     => json_encode($payload),
                    CURLOPT_RETURNTRANSFER => true,
                ]);

                $responseRaw = curl_exec($ch);

                if ($responseRaw === false) {
                    throw new Exception(curl_error($ch));
                }

                curl_close($ch);
                $responseData = json_decode($responseRaw, true);

                if (! empty($responseData['status'])) {

                    $data     = $responseData['data'] ?? [];
                    $savePath = 'assets/poduploads/';
                    if (! empty($data['docketPdfLink'])) {
                        $this->saveRemoteFile(
                            $data['docketPdfLink'],
                            $savePath . $orderId . '_' . $data['docketNo'] . '_Docket.pdf'
                        );
                    }
                    if (! empty($data['labelsLink'])) {
                        $this->saveRemoteFile(
                            $data['labelsLink'],
                            $savePath . $orderId . '_' . $data['docketNo'] . '_Label.pdf'
                        );
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Instakart Exception: ' . $e->getMessage());
            } finally {
                $this->logEdiTransaction(
                    json_encode($payload),
                    $responseData,
                    $companyCode,
                    $branchCode,
                    $userId,
                    'Instakart Carrier Order Request'
                );
            }
        }
    }

    public function formatNumber($num)
    {
        $rounded = round((float) $num, 2);
        return fmod($rounded, 1) === 0.0 ? (int) $rounded : number_format($rounded, 2, '.', '');
    }

    private function saveRemoteFile(string $url, string $path): void
    {
        $content = @file_get_contents($url);
        if ($content !== false) {
            file_put_contents($path, $content);
        }
    }

    private function logEdiTransaction(string $request, array $response, string $companyCode, string $branchCode, string $userId, string $ediName): void
    {
        $this->CI->edi_logger->setEdi_type(1);
        $this->CI->edi_logger->setTransaction_id(time());
        $this->CI->edi_logger->setEdi_name($ediName);
        $this->CI->edi_logger->setBounded_type(1);
        $this->CI->edi_logger->setCompany_code($companyCode);
        $this->CI->edi_logger->setBranch_code($branchCode);
        $this->CI->edi_logger->setUser_id($userId);
        $this->CI->edi_logger->setEdi_format_type('JSON');
        $status = isset($response['Message']) && $response['Message'] === 'SUCCESSFUL' ? 1 : 0;
        $this->CI->edi_logger->setStatus($status);
        $this->CI->edi_logger->setEdi_request($request);
        $this->CI->edi_logger->setEdi_response(json_encode($response));
        $this->CI->edi_logger->setObj_type_name($ediName);
        $this->CI->edi_logger->saveToEdiLogs();
    }

    public function getStatus(string $docketNo, string $orderId, int $id): int
    {
        $message       = '';
        $response      = '';
        $apiResponse   = [];
        $companyCode   = $this->CI->session->userdata('company_code');
        $branchCode    = $this->CI->session->userdata('branch_code');
        $userId        = $this->CI->session->userdata('user_id');
        $chk_qry_order = $this->CI->db->get_where('tb_orders', ['id' => $id])->row_array();
        $orderId       = $chk_qry_order['order_id'];
        if ($userId == null || $userId == "") {
            $userId = $chk_qry_order['user_id'];
        }
        if ($companyCode == null || $companyCode == "") {
            $companyCode = $chk_qry_order['company_code'];
        }
        if ($branchCode == null || $branchCode == "") {
            $branchCode = $chk_qry_order['branch_code'];
        }

        try {
            $url       = $this->baseUrl;
            $serverURL = $url . 'order/track';
            $type      = "JSON";

            $data = [
                'docketNo' => (array) $docketNo,
            ];

            $curl = curl_init();
            curl_setopt_array($curl, [
                CURLOPT_URL            => $serverURL,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING       => '',
                CURLOPT_MAXREDIRS      => 10,
                CURLOPT_TIMEOUT        => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST  => 'POST',
                CURLOPT_POSTFIELDS     => json_encode($data),
                CURLOPT_HTTPHEADER     => [
                    'Content-Type: application/json',
                    "Authorization: Bearer $this->appKey",
                ],
            ]);

            $response = curl_exec($curl);

            curl_close($curl);

            $data = json_decode($response, true);

            if ($data['status'] != 1) {
                return $apiResponse;
            }

            $apiResponse = json_decode($response, true);
            if ($apiResponse['status'] && isset($apiResponse['data'])) {
                $podImageUrl        = $apiResponse['data'][0]['podImageUrl'] ?? "";
                $edd                = $apiResponse['data'][0]['edd'] ?? "";
                $check_array_status = [
                    'order_id'    => $orderId,
                    'status_code' => 'SV0510',
                    'status'      => 1,
                ];
                $chk_qry_stop = $this->CI->db->select("id")->get_where('tb_stop_status', $check_array_status);
                if ($chk_qry_stop->num_rows() == 0) {
                    if (! empty($podImageUrl)) {
                        $podInput = '<SVKDocumentUpload>';
                        $podInput .= '<SVKHeader>';
                        $podInput .= '<SVKSourceApp>' . $carrier . '</SVKSourceApp>';
                        $podInput .= '<SVKDestinationApp>SVKonekt</SVKDestinationApp>';
                        $podInput .= '</SVKHeader>';
                        $podInput .= '<SVKAction>POD UPLOADED</SVKAction>';
                        $podInput .= '<SVKOrderID>' . $orderId . '</SVKOrderID>';
                        $podInput .= '<SVKDocumentType>POD</SVKDocumentType>';
                        $podInput .= '<SVKUploadDetails>';
                        $podInput .= '<SVKDocumentName>POD</SVKDocumentName>';
                        $podInput .= '<SVKURL>' . $podImageUrl . '</SVKURL>';
                        $podInput .= '</SVKUploadDetails>';
                        $podInput .= '</SVKDocumentUpload>';

                        $xml = simplexml_load_string($podInput);
                        $res = $this->CI->flipkartediservices->getXPPod($xml, $id);
                    }

                    if (! empty($edd)) {

                        $utcTz = new DateTimeZone('UTC');
                        $dt    = DateTime::createFromFormat('d/m/Y', $edd, $utcTz);

                        if ($dt) {

                            $startDatetimeUtc = $dt->setTime(0, 0, 0)->format('Y-m-d H:i:s');

                            $endDatetimeUtc = $dt->setTime(23, 59, 59)->format('Y-m-d H:i:s');

                            $status = $this->CI->common->updatetbledata(
                                "tb_orders",
                                [
                                    'delivery_datetime' => $startDatetimeUtc,
                                    'drop_endtime'      => $endDatetimeUtc,
                                    'updatedon'         => gmdate('Y-m-d H:i:s'),
                                ],
                                ['id' => $id]
                            );
                        } else {
                            log_message('error', "Invalid EDD format: " . $edd);
                        }
                    }

                    $statusList = $apiResponse['data'][0]['eventList'];

                    $statusInput = $this->generateStatusXML($docketNo, $statusList);

                    libxml_use_internal_errors(true);

                    $xml = simplexml_load_string($statusInput);

                    if ($xml === false) {
                        log_message('error', 'Invalid Status XML');
                        return 0;
                    }

                    return $this->getStatusXml($xml, $id, 'FLIPKART');
                }
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
        } finally {
            $this->CI->edi_logger->setEdi_type(1);
            $this->CI->edi_logger->setTransaction_id(time());
            $this->CI->edi_logger->setEdi_name('Instakart Carrier Tracking Status');
            $this->CI->edi_logger->setBounded_type(1);
            $this->CI->edi_logger->setCompany_code($companyCode);
            $this->CI->edi_logger->setBranch_code($branchCode);
            $this->CI->edi_logger->setUser_id($userId);
            $this->CI->edi_logger->setEdi_format_type($type);
            $this->CI->edi_logger->setStatus(((isset($apiResponse) && $apiResponse) ? 1 : 0));
            $this->CI->edi_logger->setEdi_request($docketNo);
            $this->CI->edi_logger->setEdi_response(json_encode($response));
            $this->CI->edi_logger->setObj_type_name('Instakart Carrier Tracking Status');
            $this->CI->edi_logger->saveToEdiLogs();
        }
        return $apiResponse;
    }

    private function generateStatusXML(string $docketNo, array $statusList): string
    {
        $xml = '<SVKEDIResponseMessage>';
        $xml .= '<SVKEDITransmissionHeader>';
        $xml .= '<EDIVersion>SVK2.0</EDIVersion>';
        $xml .= '<AckSpec><AckOption>SUCCESS</AckOption></AckSpec>';
        $xml .= '<SourceApp>FLIPKART</SourceApp>';
        $xml .= '<DestinationApp>SVKONEKT</DestinationApp>';
        $xml .= '<Action>BookingStatusUpdate</Action>';
        $xml .= '</SVKEDITransmissionHeader>';
        $xml .= '<SVKEDITransmissionBody><Order>';
        $xml .= '<OrderID>' . $docketNo . '</OrderID>';

        foreach ($statusList as $status) {
            $xml .= '<Status>';
            $xml .= '<StatusCode>' . ($status['statusCode'] ?? '') . '</StatusCode>';
            $xml .= '<StatusValue>' . ($status['eventDescription'] ?? '') . '</StatusValue>';
            $xml .= '<StatusType>' . ($status['statusCode'] ?? '') . '</StatusType>';
            $xml .= '<DateTime>' . ($status['eventDateTime'] ?? '') . '</DateTime>';
            $xml .= '<TimeZone>+00:00</TimeZone>';
            $xml .= '<Location>' . ($status['eventLocation'] ?? '') . '</Location>';
            $xml .= '</Status>';
        }

        $xml .= '</Order></SVKEDITransmissionBody></SVKEDIResponseMessage>';
        return $xml;
    }

    public function getStatusXml(object $xmlData, int $orderId, string $carrier): int
    {
        $statusResponse = 0;

        $body = $xmlData->SVKEDITransmissionBody ?? null;
        if (! $body || empty($body->Order)) {
            log_message('error', 'SVKEDITransmissionBody or Order missing');
            return 0;
        }

        $orders = is_array($body->Order) || $body->Order instanceof Traversable
            ? $body->Order
            : [$body->Order];

        foreach ($orders as $orderNode) {

            $externalOrderId = (string) ($orderNode->OrderID ?? '');
            if ($externalOrderId === '') {
                continue;
            }

            $chkOrder = $this->CI->db
                ->get_where('tb_orders', ['id' => $orderId])
                ->row_array();

            if (! $chkOrder) {
                log_message('error', 'Order not found in DB: ' . $orderId);
                continue;
            }

            $orderIdDb  = $chkOrder['order_id'];
            $ordersData = [];

            if (empty($orderNode->Status)) {
                continue;
            }

            $statuses = is_array($orderNode->Status) || $orderNode->Status instanceof Traversable
                ? $orderNode->Status
                : [$orderNode->Status];

            foreach ($statuses as $status) {

                $statusCode = (string) ($status->StatusCode ?? '');
                if ($statusCode === '') {
                    continue;
                }

                $ordersData[] = [
                    "OrderID"     => $orderIdDb,
                    "statustype"  => (string) ($status->StatusType ?? ''),
                    "statuscode"  => $statusCode,
                    "statusvalue" => (string) ($status->StatusValue ?? ''),
                    "datetime"    => (string) ($status->DateTime ?? ''),
                    "timezone"    => (string) ($status->TimeZone ?? ''),
                    "lat"         => (string) ($status->Lat ?? ''),
                    "lng"         => (string) ($status->Lng ?? ''),
                    "location"    => (string) ($status->Location ?? ''),
                ];
            }

            if (! empty($ordersData)) {
                $statusResponse = $this->CI
                    ->Singlestatusmodel
                    ->updateXPStatus(json_encode($ordersData), $carrier);
            }
        }

        return $statusResponse;
    }
}
