<?php
defined('BASEPATH') or exit('No direct script access allowed');

class LogyXpressEDIServices
{
    private $CI;
    private $carrier = 'LOGYXPRESS';
    private $appKey  = '';
    private $baseUrl = '';
    
    public function __construct()
    {
        $this->CI = &get_instance();
        $this->CI->load->model(['Singlestatusmodel', 'Order']);
        $this->CI->load->library(['Edi_logger', 'uuid']);
        $this->CI->load->helper('user_helper');

        $tokenRow = $this->CI->db
            ->get_where('tb_auth_tokens', ['token_for' => 'LOGYXPS_AUTH'])
            ->row_array();

        $this->appKey  = $tokenRow['auth_token'] ?? '';
        $this->baseUrl = $tokenRow['base_url'] ?? '';
        $this->code    = $tokenRow['code'] ?? '';
    }
    
    public function getProductRequestId(array $productRequest): string
    {
        $vendor   = $this->CI->db->get_where("tb_vendors", ['code' => $this->code])->row_array();

        $companyCode = $vendor['company_code'] ?? '';
        $branchCode  = $vendor['branch_code'] ?? '';
        $userId      = $vendor['user_id'] ?? '';

        try {
             
            $jsonData = json_encode($productRequest);

            log_message("error", "ProductRequest JSON: " . $jsonData);

            $url         = $this->baseUrl . "catalogue/create-single";
           
            $ch = curl_init($url);

            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Content-Type: application/json",
                "Authorization: Bearer " . $this->appKey,
                "Accept: application/json",
            ]);

            log_message("error", "AuthToken:" . $this->appKey);

            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

            // Execute the request
            $productResponse = curl_exec($ch);
            log_message("error", "Response:" . json_encode($productResponse));

            $productResponseData = json_decode($productResponse, true);

            $productId = $productResponseData['catalogue']['_id'];

            log_message("error", "Product Id: " . $productId);

            curl_close($ch);

        } catch (Exception $e) {
            log_message("error", "LogyXpress Order Exception: " . $e->getMessage());
        } finally {
            $this->logEdiTransaction($jsonData, $productResponseData, $companyCode, $branchCode, $userId, 'LogyXpress Carrier Order Request');
        }
        return $productId;
    }
    
    public function getPickupId(array $logyxpsPickupRequest): string
    {
        $vendor   = $this->CI->db->get_where("tb_vendors", ['code' => $this->code])->row_array();

        $companyCode = $vendor['company_code'] ?? '';
        $branchCode  = $vendor['branch_code'] ?? '';
        $userId      = $vendor['user_id'] ?? '';

        try {
            log_message("error","logyxpsPickupRequest:".json_encode($logyxpsPickupRequest));
                
            $logyxpsPickupRequestData = json_encode($logyxpsPickupRequest);
    
            log_message("error", "ProductRequest JSON: " . $logyxpsPickupRequestData);
            
            $url         = $this->baseUrl . "add-pickup-address";
               
            $ch = curl_init($url);
    
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Content-Type: application/json",
                "Authorization: Bearer " . $this->appKey,
                "Accept: application/json",
            ]);
    
            log_message("error", "AuthToken:" . $this->appKey);
    
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $logyxpsPickupRequestData);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
            $pickupResponse = curl_exec($ch);
            log_message("error", "Response:" . json_encode($pickupResponse));
    
            $pickupResponseData = json_decode($pickupResponse, true);
    
            $pickupId = $pickupResponseData['pickup_address']['_id'];
    
            curl_close($ch);

        } catch (Exception $e) {
            log_message("error", "LogyXpress Order Exception: " . $e->getMessage());
        } finally {
            $this->logEdiTransaction($logyxpsPickupRequestData, $pickupResponseData, $companyCode, $branchCode, $userId, 'LogyXpress Carrier Order Request');
        }
        return $pickupId;
    }
    
    public function createOrder(array $logyxpsRequest, int $orderId): string
    {
        $cdate                = date('Y-m-d H:i:s');
        $vendor   = $this->CI->db->get_where("tb_vendors", ['code' => $this->code])->row_array();

        $companyCode = $vendor['company_code'] ?? '';
        $branchCode  = $vendor['branch_code'] ?? '';
        $userId      = $vendor['user_id'] ?? '';

        $responseArray = [];
        
        try {
            log_message("error", "logyxpsRequest:" . json_encode($logyxpsRequest));
    
            $logyxpsRequestData = json_encode($logyxpsRequest);
    
            log_message("error", "LogyxpsRequest JSON: " . $logyxpsRequestData);
            
            $url         = $this->baseUrl . "orders/add-order";
    
            $ch = curl_init($url);
    
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Content-Type: application/json",
                "Authorization: Bearer " . $this->appKey,
                "Accept: application/json",
            ]);
    
            log_message("error", "AuthToken:" . $this->appKey);
    
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $logyxpsRequestData);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
            // Execute the request
            $logyxpsResponse = curl_exec($ch);
            log_message("error", "Response:" . json_encode($logyxpsResponse));
    
            $logyxpsResponseData = json_decode($logyxpsResponse, true);
            $logyxpsId           = $logyxpsResponseData['order']['_id'];
    
            log_message("error", "Logyxps_id: " . $logyxpsId);
    
            curl_close($ch);
    
            if (! empty($logyxpsId) && $logyxpsId !== '') {
    
                $logyxps_tkn_ref = [
                    'order_id'     => $orderId,
                    'reference_id' => 'TKN',
                    'ref_value'    => $logyxpsId,
                    'createdon'    => $cdate,
                ];
    
                $insTkn = $this->CI->db->insert('tb_order_references', $logyxps_tkn_ref);
               
                $upd = $this->CI->db->where(['id' => $orderId])->update("tb_orders", ['external_order_id' => $logyxpsId]);
            }
            log_message("error", "LogyXpress Order Response: " . json_encode($responseArray));

        } catch (Exception $e) {
            log_message("error", "LogyXpress Order Exception: " . $e->getMessage());
        } finally {
            $this->logEdiTransaction($logyxpsRequestData, $responseArray, $companyCode, $branchCode, $userId, 'LogyXpress Carrier Order Request');
        }
        return $logyxpsResponse;
    }

    public function getStatus(string $docketNo, int $orderId): int
    {
        if (empty($docketNo) || $orderId <= 0) {
            log_message('error', "Invalid DocketNo");
            return 0;
        }

        $url = $this->baseUrl . '/TrackConsignment?' . http_build_query([
            'Status'       => 'L',
            'lstAWB'       => $docketNo,
            'customerCode' => $this->customerCode,
        ]);

        $curl = curl_init($url);
        curl_setopt_array($curl, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPGET        => true,
            CURLOPT_HTTPHEADER     => [
                "Content-Type: application/json",
                "X-APPKEY: {$this->appKey}",
            ],
            CURLOPT_TIMEOUT => 30,
        ]);

        $response = curl_exec($curl);
        curl_close($curl);

        if ($response === false) {
            log_message('error', 'Curl Error: ' . curl_error($curl));
            return 0;
        }

        $data = json_decode($response, true);
        log_message('error', "API Response: " . $response);

        $responseStatus = $data['ResponseStatus'] ?? [];
        $trackingResult = $data['TrackingResult'][0] ?? [];

        if (($responseStatus['Message'] ?? '') !== 'SUCCESS') {
            return 0;
        }

        $statusList = $trackingResult['TrackList'] ?? [];

        // Generate XML
        $statusInput = $this->generateStatusXML($docketNo, $statusList);

        // Convert XML and save status
        $xml = simplexml_load_string($statusInput);
        return $this->getStatusXml($xml, $orderId, $this->carrier);
    }

    private function generateStatusXML(string $docketNo, array $statusList): string
    {
        $xml = '<SVKEDIResponseMessage>';
        $xml .= '<SVKEDITransmissionHeader>';
        $xml .= '<EDIVersion>SVK2.0</EDIVersion>';
        $xml .= '<AckSpec><AckOption>SUCCESS</AckOption></AckSpec>';
        $xml .= '<SourceApp>' . $this->carrier . '</SourceApp>';
        $xml .= '<DestinationApp>SVKONEKT</DestinationApp>';
        $xml .= '<Action>BookingStatusUpdate</Action>';
        $xml .= '</SVKEDITransmissionHeader>';
        $xml .= '<SVKEDITransmissionBody><Order>';
        $xml .= '<OrderID>' . $docketNo . '</OrderID>';

        foreach ($statusList as $status) {
            $status['ActivityCode'] = "BOK";
            $status['ActivityDate'] = DateTime::createFromFormat('d/m/Y H:i:s', $status['ActivityDate'])->format('Y-m-d H:i:s');
            $xml .= '<Status>';
            $xml .= '<StatusCode>' . ($status['ActivityCode'] ?? '') . '</StatusCode>';
            $xml .= '<StatusValue>' . ($status['ActivityCode'] ?? '') . '</StatusValue>';
            $xml .= '<StatusType>' . ($status['ActivityCode'] ?? '') . '</StatusType>';
            $xml .= '<DateTime>' . ($status['ActivityDate'] ?? '') . '</DateTime>';
            $xml .= '<TimeZone>+00:00</TimeZone>';
            $xml .= '<Location>' . ($status['ActivityFromLocation'] ?? '') . '</Location>';
            $xml .= '</Status>';
        }

        $xml .= '</Order></SVKEDITransmissionBody></SVKEDIResponseMessage>';

        log_message('error', "Generated Status XML: " . $xml);
        return $xml;
    }

    private function logEdiTransaction(string $request, array $response, string $companyCode, string $branchCode, string $userId, string $ediName): void
    {
        $this->CI->edi_logger->setEdi_type(1);
        $this->CI->edi_logger->setTransaction_id(time());
        $this->CI->edi_logger->setEdi_name($ediName);
        $this->CI->edi_logger->setBounded_type(1);
        $this->CI->edi_logger->setCompany_code($companyCode);
        $this->CI->edi_logger->setBranch_code($branchCode);
        $this->CI->edi_logger->setUser_id($userId);
        $this->CI->edi_logger->setEdi_format_type('JSON');
        $status = isset($response['ResponseStatus']['Message']) && $response['ResponseStatus']['Message'] === 'SUCCESS' ? 1 : 0;
        $this->CI->edi_logger->setStatus($status);
        $this->CI->edi_logger->setEdi_request($request);
        $this->CI->edi_logger->setEdi_response(json_encode($response));
        $this->CI->edi_logger->setObj_type_name($ediName);
        $this->CI->edi_logger->saveToEdiLogs();
    }

    public function getStatusXml(object $xmlData, int $orderId, string $carrier): int
    {
        $statusResponse = 0;
        $body           = $xmlData->xpath('SVKEDITransmissionBody');

        if (empty($body)) {
            return 0;
        }

        foreach ($body[0]->Order as $order) {
            $chkOrder = $this->CI->db->get_where('tb_orders', ['id' => $orderId])->row_array();
            if (! $chkOrder) {
                continue;
            }

            $ordersData = [];
            $orderIdDb  = $chkOrder['order_id'];

            foreach ($order->Status as $status) {
                $statusCode = (string) ($status->StatusCode ?? '');
                if (! $statusCode) {
                    continue;
                }

                $ordersData[] = [
                    "OrderID"     => $orderIdDb,
                    "statustype"  => (string) ($status->StatusType ?? ''),
                    "statuscode"  => $statusCode,
                    "statusvalue" => (string) ($status->StatusValue ?? ''),
                    "datetime"    => (string) ($status->DateTime ?? ''),
                    "timezone"    => (string) ($status->TimeZone ?? ''),
                    "lat"         => (string) ($status->Lat ?? ''),
                    "lng"         => (string) ($status->Lng ?? ''),
                    "location"    => (string) ($status->Location ?? ''),
                ];
            }

            if (! empty($ordersData)) {
                $statusResponse = $this->CI->Singlestatusmodel->updateXPStatus(json_encode($ordersData), $carrier);
            }
        }

        return $statusResponse;
    }
}
