<?php

use phpseclib\Net\SFTP;

/*
 *  Please refer Vendorservice library to know more about how to use this library.
 */

class CarrierNotificationSftpManager
{
    private $shipmentid = 0;
    private $url;
    private $username;
    private $password;
    private $path;
    private $backupPath;

    public function getUrl(): string
    {
        return $this->url;
    }

    public function setUrl(string $url): void
    {
        $this->url = $url;
    }

    public function getUsername(): string
    {
        return $this->username;
    }

    public function setUsername(string $username): void
    {
        $this->username = $username;
    }

    public function getPassword(): string
    {
        return $this->password;
    }

    public function setPassword(string $password): void
    {
        $this->password = $password;
    }

    public function getPath(): string
    {
        return $this->path;
    }

    public function setPath(string $path): void
    {
        $this->path = $path;
    }

    public function getBackupPath(): string
    {
        return $this->backupPath;
    }

    public function setBackupPath(string $backupPath): void
    {
        $this->backupPath = $backupPath;
    }

    public function setShipmentid(int $shipmentid): void
    {
        $this->shipmentid = $shipmentid;
    }

    public function getShipmentid(): int
    {
        return $this->shipmentid;
    }

    public function uploadToSftp(string $xmlContent, string $filename): void
    {
        $sftp = $this->getSftpInstance();
        if (!$sftp) {
            self::throwException(
                'Unable to login in SFTP server/' . $this->getUrl() . '/with user /' . $this->getUsername()
            );
        }
        try {
            $sftp->chdir($this->getPath());
        } catch (\Exception $exception) {
            self::throwException(
                'Unable to go to directory /' . $this->getPath() . '/ in SFTP server/' . $this->getUrl(
                ) . '/ with user /' . $this->getUsername() . '/' . $exception->getMessage()
            );
        }
        try {
            $sftp->put($filename . '.xml', $xmlContent, SFTP::SOURCE_STRING);
        } catch (\Exception $exception) {
            self::throwException(
                "Unable to upload " . $filename . '.xml' . '/ to directory /' . $this->getPath(
                ) . ' in SFTP server ' . $this->getUrl() . '/ with user /' . $this->getUsername(
                ) . '/' . $exception->getMessage()
            );
        }

        if ($this->getBackupPath() !== false && strlen(trim($this->getBackupPath())) > 6) {
            try {
                $sftp->chdir($this->getBackupPath());
            } catch (\Exception $exception) {
                self::throwException(
                    'Unable to go to directory /' . $this->getBackupPath() . '/ in SFTP server/' . $this->getUrl(
                    ) . '/ with user /' . $this->getUsername() . '/' . $exception->getMessage()
                );
            }
            try {
                $sftp->put($filename . '.xml', $xmlContent, SFTP::SOURCE_STRING);
            } catch (\Exception $exception) {
                self::throwException(
                    'Unable to upload ' . $filename . '.xml' . '/to directory /' . $this->getBackupPath(
                    ) . ' in SFTP server ' . $this->getUrl() . '/ with user /' . $this->getUsername(
                    ) . '/' . $exception->getMessage()
                );
            }
        }
        unset($sftp);
    }

    private static function throwException(string $message): void
    {
        $ci = &get_instance();
        $ci->load->helper('log');
        log_error($message, 2);
        throw new \RuntimeException($message);
    }

    public function getSftpInstance()
    {
        $sftp = new SFTP($this->getUrl());
        if ($sftp->login($this->getUsername(), $this->getPassword())) {
            return $sftp;
        }
        unset($sftp);
        return false;
    }

    public function initialLoading(): void
    {
        // check sftp connection
        $response = ['ConnectionStatus' => 'SFTP CONNECTION FAIL', 'FOLDER' => 'NO FOLDER FOUND'];
        $sftp = new SFTP($this->getUrl());
        if ($sftp->login($this->getUsername(), $this->getPassword())) {
            $response['ConnectionStatus'] = '  CONNECTION SUCCESS';
        }
        // folders list
        $files = $sftp->nlist($this->getPath());
        if ($files) {
            $response['FOLDER'] = $files;
        }
    }

    public static function generateSftpFolderName($carrier_name): string
    {
        return str_replace(' ', '', strtolower($carrier_name)) . '_edi';
    }

    public function initializeConnectionSettings(array $connectionProperties): void
    {
        if (empty($connectionProperties)) {
            self::throwException('Please provide valid connection properties');
        }

        if (isset($connectionProperties['url']) && strlen(trim($connectionProperties['url'])) <= 2) {
            self::throwException('Please provide valid connection properties');
        }

        if (isset($connectionProperties['username']) && strlen(trim($connectionProperties['username'])) <= 2) {
            self::throwException('Please provide valid connection properties');
        }

        if (isset($connectionProperties['password']) && strlen(trim($connectionProperties['password'])) <= 2) {
            self::throwException('Please provide valid connection properties');
        }

        if (isset($connectionProperties['path']) && strlen(trim($connectionProperties['path'])) <= 2) {
            self::throwException('Please provide valid connection properties');
        }

        $this->setUrl($connectionProperties['url']);
        $this->setUsername($connectionProperties['username']);
        $this->setPassword($connectionProperties['password']);
        $this->setPath($connectionProperties['path']);
        $this->setBackupPath($connectionProperties['backupPath'] ?? false);
    }
}
