<?php


/**
 * @property CI_Controller $ci
 */
class Cronjob_runner
{
    /** @var string No DI, but sort of ready for DI. If we ever get it */
    private const CODEIGNITER_CLI_PATH = '/var/www/html/index.php';

    /** @var string The amount of memory the cronjob can */
    private const MEMORY_LIMIT = '4096M';

    /**
     *
     */
    public function __construct()
    {
        $this->ci = &get_instance();
        $this->ci->load->library('email');
    }

    /**
     * @param string $cronjobRootName
     * @param string $cronFunction
     *
     * A function to run the cronjob a-sync. The goal is to run the cronjob in the background
     * while giving AWS Lambda a http 200 status.
     * The cronjob should be able to use more memory than a normal web call. And it should be able
     * to run for a longer period than the normal web call.
     */
    public function runCronjob(string $cronjobRootName, string $cronFunction): void
    {
        $this->runCron($cronjobRootName, $cronFunction);
        exit();
    }

    public function runLockedCronjob(string $cronjobRootName, string $cronFunction): void
    {
        $filename = preg_replace("/[^a-zA-Z0-9]+/", "", $cronjobRootName . $cronFunction) . '.lock';

        if (file_exists(__DIR__ . "/" . $filename)) {
            $fileCreatedTime = filemtime(__DIR__ . "/" . $filename);

            if ($fileCreatedTime < (time() - (60 * 60 * 6))) {
                $this->sendLockFailEmail($cronjobRootName, $cronFunction, $filename);
            }

            return;
        }

        $fp = fopen(__DIR__ . "/" . $filename, "w+");
        if (flock($fp, LOCK_EX | LOCK_NB)) {
            fwrite($fp, time());
            $pid = pcntl_fork();
            if ($pid === 0) {
                $this->runCron($cronjobRootName, $cronFunction);
                flock($fp, LOCK_UN);
                unlink(__DIR__ . "/" . $filename);
                exit();
            }
        }
        fclose($fp);
    }

    /**
     * @param string $cronjobRootName
     * @param string $cronFunction
     * @param string $filename
     */
    private function sendLockFailEmail(string $cronjobRootName, string $cronFunction, string $filename): void
    {
        $this->ci->email->to('svkonekt-tech-dev-ops-cloud@kuehne-nagel.com');
        $this->ci->email->from('svkonekt@kuehne-nagel.com', 'svkonekt');
        $this->ci->email->subject(sprintf('It looks like cronjob %s %s has been running for more then 6 hours', $cronjobRootName, $cronFunction));
        $this->ci->email->set_mailtype('text');
        $this->ci->email->message(sprintf('Please investigate if the %s %s cron is still running. Or that the lock wasn\'t removed successfully. Please remove %s lock file if needed.',
            $cronjobRootName,
            $cronFunction,
            __DIR__ . "/" . $filename
        ));
        $this->ci->email->send();
        $this->ci->email->clear(true);
    }

    /**
     * @param string $cronjobRootName
     * @param string $cronFunction
     */
    private function runCron(string $cronjobRootName, string $cronFunction): void
    {
         exec(
            sprintf(
                '/usr/local/bin/php -d memory_limit=%s %s %s %s > /dev/null 2>&1 &',
                static::MEMORY_LIMIT,
                static::CODEIGNITER_CLI_PATH,
                $cronjobRootName,
                $cronFunction
            ),
             $output
        );

        foreach($output as $logEntry) {
            if (!empty($logEntry)) {
                log_message('error', print_r($logEntry, true));
            }
        }
    }
}
