<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AlertsModel extends CI_Model 
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get customer delay data with proper filtering and business logic
     */
    public function getCustomerDelayData($post = array(), $generalArray = array()):array
    {
        $whr = "";
        $bindings = array();
        
        // DEBUG: Log the incoming dates
        log_message('debug', 'AlertsModel - Search dates - From: ' . ($post['aqufromdate'] ?? 'empty') . ', To: ' . ($post['aqutodate'] ?? 'empty'));
        
        // Add date filters if provided
        if (!empty($post['aqufromdate'])) {
            $whr .= " AND DATE(o.createdon) >= ?";
            $bindings[] = $post['aqufromdate'];
        }
        if (!empty($post['aqutodate'])) {
            $whr .= " AND DATE(o.createdon) <= ?";
            $bindings[] = $post['aqutodate'];
        }

        if (!empty($post['order_id'])) {
            $whr .= " AND o.order_id = ?";
            $bindings[] = $post['order_id'];
        }

        // Apply user context filtering
        if (!empty($generalArray)) {
            $userid = $generalArray['userid'];
            $custid = $generalArray['custid'];
            $businessType = $generalArray['business_type'];
            $countryUserids = $generalArray['countryUserids'];
            $subcusts = $generalArray['subcusts'];
            $vendorId = $generalArray['vendor_id'];

            // Apply user filtering
            if ($userid != 0) {
                if (!empty($countryUserids)) {
                    $countryIdsPlaceholder = str_repeat('?,', count($countryUserids) - 1) . '?';
                    $whr .= " AND o.user_id IN ($countryIdsPlaceholder)";
                    $bindings = array_merge($bindings, $countryUserids);
                } else {
                    $whr .= " AND o.user_id = ?";
                    $bindings[] = $userid;
                }
            }

            // Apply customer filtering
            if (!empty($subcusts)) {
                $subcustPlaceholder = str_repeat('?,', count($subcusts) - 1) . '?';
                $whr .= " AND o.customer_id IN ($subcustPlaceholder)";
                $bindings = array_merge($bindings, $subcusts);
            } else if (!empty($custid)) {
                $whr .= " AND o.customer_id = ?";
                $bindings[] = $custid;
            }

            // Apply vendor filtering if applicable
            if (!empty($vendorId) && $businessType != "Customer") {
                $whr .= " AND o.vendor_id = ?";
                $bindings[] = $vendorId;
            }
        }
        // Filter by delay type if set
        if (!empty($post['delay']) && $post['delay'] != 'both') {
            if ($post['delay'] === 'pickup') {
                $whr .= " AND (
                    NOW() > o.pickup_datetime
                    AND NOT EXISTS (
                        SELECT 1 FROM tb_stop_status ss
                        WHERE ss.shipment_id = o.shift_id 
                        AND ss.status_code IN ('0500','SV0131')
                    )
                )";
            } elseif ($post['delay'] === 'delivery') {
                $whr .= " AND (
                    NOW() > o.delivery_datetime
                    AND EXISTS (
                        SELECT 1 FROM tb_stop_status ss
                        WHERE ss.shipment_id = o.shift_id
                        AND ss.status_code IN ('0500','SV0131')
                    )
                    AND NOT EXISTS (
                        SELECT 1 FROM tb_stop_status ss
                        WHERE ss.shipment_id = o.shift_id
                        AND ss.status_code IN ('3000','SV0510')
                    )
                )";
            }
        }
        return $this->executeDelayQuery($whr, $bindings);
    }

    /**
     * Execute the main delay query
     */
    private function executeDelayQuery($whr, $bindings):array
    {
        // FIRST: Check if there are ANY orders in the date range (for debugging)
        $countQuery = "
            SELECT COUNT(*) as total_orders
            FROM tb_orders o
            WHERE o.status != 0
                $whr
        ";
        
        $countResult = $this->db->query($countQuery, $bindings);
        $totalOrders = $countResult->row()->total_orders;
        log_message('debug', 'AlertsModel - Total orders in date range: ' . $totalOrders);

        // MAIN QUERY with all conditions
        $query = "
            SELECT 
                o.order_id,
                o.shipmentid,
                o.pickup_datetime,
                o.delivery_datetime,
                o.pickup_address1,
                o.delivery_address1,
                o.createdon,
                o.trip_id,
                o.shift_id,
                v.name AS carrier,
                sm.status_name AS latest_status_name,
                s.createdon AS latest_createdon,
                COALESCE(trip.register_number, shift.register_number) AS vehicle_register_number,
                COALESCE(trip.trucktype, shift.trucktype) AS truck_type,
                COALESCE(trip.drivername, shift.drivername) AS driver_name,
                CASE
                    WHEN NOW() > o.pickup_datetime 
                         AND NOT EXISTS (
                             SELECT 1 
                             FROM tb_stop_status ss
                             WHERE ss.shipment_id = o.shift_id
                               AND ss.status_code IN ('0500','SV0131')
                         ) THEN 'Pickup Delay'
                    WHEN NOW() > o.delivery_datetime 
                         AND EXISTS (
                             SELECT 1
                             FROM tb_stop_status ss
                             WHERE ss.shipment_id = o.shift_id
                               AND ss.status_code IN ('0500','SV0131')
                         )
                         AND NOT EXISTS (
                             SELECT 1 
                             FROM tb_stop_status ss
                             WHERE ss.shipment_id = o.shift_id
                               AND ss.status_code IN ('3000','SV0510')
                         ) THEN 'Delivery Delay'
                    ELSE 'No Delay'
                END AS remark
            FROM tb_orders o
            LEFT JOIN (
                SELECT *
                FROM (
                    SELECT *,
                           ROW_NUMBER() OVER (PARTITION BY shipment_id ORDER BY createdon DESC, id DESC) AS rn
                    FROM tb_stop_status
                    WHERE shipment_id > 0
                ) ranked
                WHERE rn = 1
            ) s ON s.shipment_id = o.shift_id
            LEFT JOIN (
                SELECT status_code, status_name
                FROM (
                    SELECT *,
                           ROW_NUMBER() OVER (PARTITION BY status_code ORDER BY createdon DESC) AS rn
                    FROM tb_status_master
                ) sm_ranked
                WHERE rn = 1
            ) sm ON sm.status_code = s.status_code
            LEFT JOIN tb_vendors v ON v.id = o.vendor_id
            LEFT JOIN (
                SELECT t.shift_id, td.register_number, ttt.trucktype, dr.name AS drivername
                FROM tb_trips t
                LEFT JOIN tb_trucks_data td ON t.vehicle_id = td.id
                LEFT JOIN tb_trucktypes ttt ON ttt.id = td.truck_type
                LEFT JOIN tb_truck_drivers dr ON t.driver_id = dr.id
                GROUP BY t.shift_id
            ) trip ON trip.shift_id = o.shift_id
            LEFT JOIN (
                SELECT s.id AS shift_id, td.register_number, ttt.trucktype, dr.name AS drivername
                FROM tb_shifts s
                LEFT JOIN tb_shft_veh sv ON sv.shft_id = s.id
                LEFT JOIN tbl_assigned_drivers ad ON ad.vehicle_id = sv.vehicle_id AND ad.status ='1'
                LEFT JOIN tb_trucks_data td ON td.id = sv.vehicle_id
                LEFT JOIN tb_trucktypes ttt ON ttt.id = td.truck_type
                LEFT JOIN tb_truck_drivers dr ON dr.id = ad.driver_id
                GROUP BY s.id
            ) shift ON shift.shift_id = o.shift_id
            WHERE 
                o.trip_id > 0
                AND o.shift_id > 0
                AND o.shipmentid IS NOT NULL
                AND o.status != 0
                AND (
                    (NOW() > o.pickup_datetime AND NOT EXISTS (
                        SELECT 1 
                        FROM tb_stop_status ss
                        WHERE ss.shipment_id = o.shift_id
                          AND ss.status_code IN ('0500','SV0131')
                    ))
                 OR (NOW() > o.delivery_datetime 
                     AND EXISTS (
                         SELECT 1
                         FROM tb_stop_status ss
                         WHERE ss.shipment_id = o.shift_id
                           AND ss.status_code IN ('0500','SV0131')
                     )
                     AND NOT EXISTS (
                         SELECT 1 
                         FROM tb_stop_status ss
                         WHERE ss.shipment_id = o.shift_id
                           AND ss.status_code IN ('3000','SV0510')
                     )
                 )
                )
                $whr
            ORDER BY o.order_id DESC, o.shipmentid ASC
        ";
        
        // DEBUG: Log the final query
        log_message('debug', 'AlertsModel - Final query: ' . $query);
        log_message('debug', 'AlertsModel - Bindings: ' . print_r($bindings, true));
        
        $result = $this->db->query($query, $bindings);
        $finalResults = $result->result_array();
        
        log_message('debug', 'AlertsModel - Final results count: ' . count($finalResults));
        
        return $finalResults;
    }

    /**
     * Validate date range (check if it exceeds maximum allowed days)
     */
    public function validateDateRange($fromDate, $toDate, $maxDays = 45):bool
    {
        if (empty($fromDate) || empty($toDate)) {
            return true; // Allow empty dates
        }

        try {
            $from = new DateTime($fromDate);
            $to = new DateTime($toDate);
            $days = $from->diff($to)->days+1;
            
            log_message('debug', 'AlertsModel - Date validation - Days difference: ' . $days);
            
            return $days <= $maxDays;
        } catch (Exception $e) {
            log_message('error', 'AlertsModel - Date validation error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get count of delay orders for dashboard
     */
    public function getDelayCount($generalArray = array()):int
    {
        $delayData = $this->getCustomerDelayData(array(), $generalArray);
        return count($delayData);
    }

    /**
     * Convert date format from MM-DD-YYYY to YYYY-MM-DD
     */
    public function convertDateFormat($dateString):string
    {
        if (empty($dateString)) {
            return '';
        }
        
        // If already in YYYY-MM-DD format, return as is
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $dateString)) {
            return $dateString;
        }
        
        // Convert MM-DD-YYYY to YYYY-MM-DD
        if (preg_match('/^(\d{1,2})-(\d{1,2})-(\d{4})$/', $dateString, $matches)) {
            $month = str_pad($matches[1], 2, '0', STR_PAD_LEFT);
            $day   = str_pad($matches[2], 2, '0', STR_PAD_LEFT);
            $year  = $matches[3];
            return "$year-$month-$day";
        }
        
        return $dateString;
    }
}
