<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Document Sequence Model.
 *
 * The goal for this model is for generating an unique sequence based on giving values.
 *
 * For example:
 * - TEST & UAT:
 *   - ETNFR1211220000001 exists of prefix [ETN] countryCode [FR] documentType [1] year [21] month [12] environement [2] sequence [0000001]
 * - PROD:
 *   - ETNFR121120000001 exists of prefix [ETN] countryCode [FR] documentType [1] year [21] month [12] sequence [0000001]
 *
 * @property CI_DB_driver $db
 */
class DocumentSequenceModel extends CI_Model
{
    const DOCUMENT_TYPE_POD = 1;
    const DOCUMENT_TYPE_POP = 2;
    const DOCUMENT_TYPE_PICTURE = 3;
    const DOCUMENT_TYPES = [self::DOCUMENT_TYPE_POD, self::DOCUMENT_TYPE_POP, self::DOCUMENT_TYPE_PICTURE];

    const ENVIRONMENT_LIVE = 0;
    const ENVIRONMENT_DEMO = 1;
    const ENVIRONMENT_UAT = 2;
    const ENVIRONMENT_DEV = 3;

    const TABLE_NAME = 'document_sequence';

    public function __construct()
    {
        parent::__construct();
    }

    public function getSequence(string $prefix, string $countryCode, int $documentType, \DateTime $dateTime, int $environmentIdentifier): string
    {
        if (self::ENVIRONMENT_LIVE === $environmentIdentifier) {
            return strtoupper(
                sprintf(
                    '%s%s%s%s%s%07d',
                    $prefix,
                    $countryCode,
                    $documentType,
                    $dateTime->format('y'),
                    $dateTime->format('m'),
                    $this->getSequenceNumber($prefix, $countryCode, $documentType, $dateTime, $environmentIdentifier)
                )
            );
        } else {
            return strtoupper(
                sprintf(
                    '%s%s%s%s%s%s%07d',
                    $prefix,
                    $countryCode,
                    $documentType,
                    $dateTime->format('y'),
                    $dateTime->format('m'),
                    $environmentIdentifier,
                    $this->getSequenceNumber($prefix, $countryCode, $documentType, $dateTime, $environmentIdentifier)
                )
            );
        }
    }

    public function getSequenceNumber(string $prefix, string $countryCode, int $documentType, \DateTime $dateTime, int $environment): int
    {
        $where = [
            'prefix' => $prefix,
            'country_code' => $countryCode,
            'document_type' => $documentType,
            'year' => (int)$dateTime->format('y'),
            'month' => (int)$dateTime->format('m'),
            'environment' => $environment
        ];

        $this->db->trans_start();

        $this->db->query(
            'INSERT INTO document_sequence (prefix, country_code, document_type, `year`, month, environment) VALUES (?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE sequence = sequence + 1;',
            array_values($where)
        );

        $sequence = $this->db->select('sequence')
            ->from(self::TABLE_NAME)
            ->where($where)->get()->row();

        $this->db->trans_complete();

        return (int)$sequence->sequence;
    }
}
