<?php
namespace App\Models;
defined('BASEPATH') or exit('No direct script access allowed');

use App\Models\Exceptions\MissingConfigurationException;
use App\Models\Exceptions\UnknownMethodException;

/**
 * Parent class for CI Models
 */
class Table extends \CI_Model {

    /**
     * Force Class to have all needed const
     */
    public function __construct()
    {
        parent::__construct();
        if ( !defined('static::TABLE') ) {
            throw new MissingConfigurationException('Please define TABLE const inside model class.');
        }
    }

    /**
     * Prepares base query with optional conditions and filtering by only active entries
     *
     * @param array $conditions
     * @param bool $active_only
     * @return \CI_DB_query_builder
     */
    public function base_query(array $conditions = [], bool $active_only = true): \CI_DB_query_builder
    {
        $this->db->reset_query();

        $query = $this->db->from(static::TABLE);
        if ($active_only) {
            $this->set_active_condition($conditions);
        }
        if (!empty($conditions)) {
            $this->_set_conditions($query, $conditions);
        }

        return $query;
    }

    /**
     * Check if row exists base on provided conditions
     *
     * @param array $conditions
     * @param bool $active_only
     * @return bool
     */
    public function exists(array $conditions = [], bool $active_only = true) : bool
    {
        return $this->count($conditions, $active_only) > 0;
    }

    /**
     * Count results base on provided conditions
     *
     * @param array $conditions
     * @param bool $active_only
     * @return int
     */
    public function count(array $conditions = [], bool $active_only = true) : int
    {
        $query = $this->base_query($conditions, $active_only);

        return $query->count_all_results();
    }

    /**
     * General finder method
     *
     * @param array $conditions
     * @param array $fields
     * @param array $options
     * @param bool $active_only
     * @return array
     * @throws Exception
     */
    public function find(array $conditions = [], array $fields = [], array $options = [], bool $active_only = true) : array
    {
        $query = $this->base_query($conditions, $active_only);

        !empty($fields) && $query = $query->select($fields);

        foreach ($options as $key => $value) {
            switch ($key) {
                case 'order_by': // 'order_by' => ['column_name' => 'ASC']
                    $query = $query->order_by($value['key'], $value['direction']);
                    break;
                case 'group_by': // 'group_by' => 'column_name'
                    $query = $query->group_by($value);
                    break;
                case 'join': // [['table' => 'table_name', 'condition' => 'a = b', 'type' => 'LEFT']]
                    foreach ($value as $join) {
                        $query = $query->join($join['table'], $join['condition'], $join['type']);
                    }
                    break;
                case 'limit': // 'limit' => 50
                    $query = $query->limit($value);
                    break;
                case 'offset': // 'offset' => 50
                    $query = $query->offset($value);
                    break;
            }
        }

        return $query->get()->result();
    }

    /**
     * Inserting provided data
     *
     * @param array $data
     * @return int
     * @throws Exception
     */
    public function insert(array $data = []) : ?int
    {
        $this->db->reset_query();

        if (empty($data)) {
            throw new MissingConfigurationException('Insert array data is empty');
        }
        if (defined('static::CREATEDON_COL') && empty($data[static::CREATEDON_COL])) {
            $data[static::CREATEDON_COL] = date('Y-m-d H:i:s');
        }

        return ($this->db->insert(static::TABLE, $data)) ? $this->db->insert_id() : null;
    }

    /**
     * Updating provided data base on conditions
     *
     * @param array $data
     * @param array $conditions
     * @return array
     * @throws Exception
     */
    public function update(array $data = [], array $conditions = []) : array
    {
        if (empty($data) || empty($conditions)) {
            throw new MissingConfigurationException('Update data or conditions are empty');
        }

        if (defined('static::CREATEDON_COL')) {
            unset($data[static::CREATEDON_COL]);
        }

        $this->db->reset_query();
        $to_update = $this->db->select(static::ID_COL)->from(static::TABLE)->where($conditions);
        $ids_to_update = array_map(
            function($row){
                return (int) $row[static::ID_COL];
            },
            $to_update->get()->result_array()
        );

        $this->db->reset_query();

        return ($this->db->where($conditions)->update(static::TABLE, $data)) ? $ids_to_update : array();
    }

    /**
     * Common function for insert/update operations
     *
     * @param array $data
     * @param array $conditions
     * @param bool $active_only
     * @return array
     * @throws Exception
     */
    public function insert_or_update(array $data = [], array $conditions = [], bool $active_only = false) : array
    {
        // if empty conditions use insert data
        empty($conditions) && $conditions = $data;

        $columns = array_keys($data);
        $columns[] = 'id';
        $exist = $this->base_query($conditions, $active_only)->select($columns)->get()->result_array();

        if (empty($exist)) {
            return [
                'ids' => array($this->insert($data, $conditions)),
                'created' => true
            ];
        }
        if (count($exist) === 1) {
            // columns that create unnecessary diff due to rounding and decimal formatting...
            $specialNumberFormatting = [
                'tb_shifts' => ['volume', 'weight'],
                'tb_orders' => ['quantity', 'volume', 'weight', 'goods_value'],
                'tb_employee' => ['shipment_weight', 'shipment_volume', 'no_of_pkgs'],
                'tb_shiporder_stops' => ['volume', 'weight'],
            ];
            // ...so force proper formatting in such cases
            if (isset($specialNumberFormatting[static::TABLE])) {
                foreach ($specialNumberFormatting[static::TABLE] as $column) {
                    if (isset($data[$column])) {
                        $data[$column] = number_format($data[$column], 2, '.', '');
                    }
                }
            }

            $exist = reset($exist);
            $diff = array_diff($data, $exist);
            if (empty($diff)) {
                return [
                    'ids' => [ (int) $exist['id'] ],
                    'created' => false
                ];
            }
        }

        return [
            'ids' => $this->update($data, $conditions),
            'created' => false
        ];
    }

    /**
     * Apply active condition inside general conditions array
     *
     * @param array $conditions
     */
    protected function set_active_condition(array &$conditions) : void
    {
        if (defined('static::STATUS_COL')) {
            $conditions[static::TABLE . '.' . static::STATUS_COL] = [1, 'Active'];
        }
    }

    /**
     * Common function to set conditions base on provided array
     *
     * @param Object $query
     * @param array $conditions
     */
    public function _set_conditions(Object &$query, array $conditions) : void
    {
        foreach ($conditions as $key => $value) {
            if (is_array($value)) {
                $query->where_in($key, $value);
            } else {
                $query->where($key, $value);
            }
        }
    }


    /**
     *
     * @param string $method
     * @param array $args
     * @return array
     */
    public function __call(string $method, array $args = []) : array
    {
        if (preg_match('/^find(?:\w+)?by/', $method) > 0) {
            return $this->_dynamic_finder($method, $args);
        }

        throw new UnknownMethodException(sprintf('Unknown method %s', $method));
    }

    /**
     * @param string $method
     * @param array $args
     * @return array
     */
    private function _dynamic_finder(string $method, array $args) : array
    {
        preg_match('/^find(?:\w+)?by_/', $method, $matches);

        if (empty($matches)) {
            throw new UnknownMethodException(sprintf('Unknown method %s', $method));
        }

        $searchField = substr($method, strlen($matches[0]));
        $fields = $args[1]['fields'] ?? [];

        $allowed = ['order_by', 'group_by', 'join', 'limit', 'offset'];
        $options = array_filter(
            $args[1] ?? [],
            function ($key) use ($allowed) {
                return in_array($key, $allowed);
            },
            ARRAY_FILTER_USE_KEY
        );

        $active_only = $args[1]['active_only'] ?? true;

        return $this->find([$searchField => $args[0]], $fields, $options, $active_only);
    }

    /**
     * @param array $conditions
     * @param bool $active_only
     * @return bool TRUE on success
     */
    public function delete(array $conditions, bool $active_only = true) : bool
    {
        $query = $this->base_query($conditions, $active_only);

        return $query->delete();
    }
}
